package logger

import (
	"fmt"
	"os"

	"code.justin.tv/devhub/mdaas-ingest/config"

	"github.com/pkg/errors"
	"github.com/rollbar/rollbar-go"
	"github.com/sirupsen/logrus"
)

func Configure(cfg config.Config, canary string) {
	logrus.SetFormatter(&logrus.JSONFormatter{})
	logrus.SetOutput(os.Stdout)
	rollbar.SetToken(cfg.Secrets.RollbarToken)
	rollbar.SetEnvironment(fmt.Sprintf("%s%s", cfg.Env, canary))
	logrus.AddHook(&rollbarErrorHook{})
}

// Implements the logrus Hook interface.
type rollbarErrorHook struct{}

type logFunc func(msg interface{})

// StandardLogger is to return the pointer to the logger object
func StandardLogger() *logrus.Logger {
	return logrus.StandardLogger()
}

func Info(msg interface{}) {
	logrus.Info(msg)
}

func Warn(msg interface{}) {
	logrus.Warn(msg)
}

func Trace(msg interface{}) {
	logrus.Trace(msg)
}

func Debug(msg interface{}) {
	logrus.Debug(msg)
}

func Error(msg interface{}) {
	logrus.Error(msg)
}

// Tailwind has a great implementation for logrus + rollbar, deriving from it https://git.xarth.tv/ce-analytics/tailwind/blob/master/logger/logger.go
// Will test out and see if their implementation meets our needs
// Logrus calls Fire when logging a message that matches
// a level provided by hook.Levels().
func (hook *rollbarErrorHook) Fire(entry *logrus.Entry) error {
	level := entry.Level
	data := map[string]interface{}(entry.Data)

	// This is tuned to show the stack trace starting with the code that called the
	// relevant logrus method. Refactoring surrounding code may require an
	// adjustment.
	stackTraceLinesToIgnore := 12

	err := errorFromEntry(entry)
	switch {
	case level == logrus.FatalLevel || level == logrus.PanicLevel:
		rollbar.Critical(err, data, stackTraceLinesToIgnore)
	case level == logrus.ErrorLevel:
		rollbar.Error(err, data, stackTraceLinesToIgnore)
	default:
		return fmt.Errorf("unexpected logging level for Rollbar logging hook: %s", level)
	}

	return nil
}

func errorFromEntry(entry *logrus.Entry) error {
	// We make an error out of entry.Message because we always want a stack trace.
	// The logrus rollbar library will only create a stack trace if there is an
	// error (and no string) passed to its error reporting methods.
	err := errors.New(entry.Message)

	// If we have an actual error, for instance from `logger.WithError(err).Error("it's bad"))`,
	// then we'll use that error and wrap it with the provided message.
	logrusErrorData := entry.Data[logrus.ErrorKey]
	logrusError, ok := logrusErrorData.(error)
	if ok {
		err = errors.Wrap(logrusError, entry.Message)
	}

	return err
}

func (hook *rollbarErrorHook) Levels() []logrus.Level {
	return []logrus.Level{
		logrus.ErrorLevel,
		logrus.FatalLevel,
		logrus.PanicLevel,
	}
}
