package models

import (
	"fmt"
)

const (
	delimiter        = "."
	gameStreamPrefix = "game"
)

// Event is a generic struct that will container either a full event or a delta
type Event struct {
	Full  *FullEvent  `json:"full,omitempty"`
	Delta *DeltaEvent `json:"delta,omitempty"`
}

// PartitionKey builds a key for event based on the broadcaster id
func (e Event) PartitionKey() string {
	if e.Full != nil {
		return e.Full.BroadcasterID
	} else if e.Delta != nil {
		return e.Delta.BroadcasterID
	}
	return ""
}

// StreamName retrieves the name of the stream where the event will be sent
func (e Event) StreamName(cloneID int) string {
	gameID := e.GetGameID()
	return getStreamName(gameID, cloneID)
}

func getStreamName(gameID string, cloneNum int) string {
	return fmt.Sprintf("%s%s%s%s%d", gameStreamPrefix, delimiter, gameID, delimiter, cloneNum)
}

// BroadcasterID just makes it less messy to pull the broadcaster id
func (e Event) BroadcasterID() string {
	if e.Full != nil {
		return e.Full.BroadcasterID
	}
	if e.Delta != nil {
		return e.Delta.BroadcasterID
	}
	return ""
}

func (e Event) GetGameID() string {
	if e.Full != nil {
		return e.Full.GameID
	}
	if e.Delta != nil {
		return e.Delta.GameID
	}
	return ""
}

func (e *Event) SetKinesisSpecificInfo(id, seqStart, firstKeyFrame string) {
	if e.Full != nil {
		full := *e.Full
		full.BroadcasterID = id
		full.FirstKeyFrame = firstKeyFrame
		e.Full = &full
	} else if e.Delta != nil {
		delta := *e.Delta
		delta.BroadcasterID = id
		delta.SeqStart = seqStart
		delta.FirstKeyFrame = firstKeyFrame
		e.Delta = &delta
	}
}
