package processor

import (
	"fmt"

	log "github.com/sirupsen/logrus"

	"code.justin.tv/devhub/twitch-e2-ingest/models"
)

func (d *DataProcessor) fanoutFullToSNS(newMetadata map[string]interface{}, refresh bool) {

	emptyMetaData := map[string]interface{}{
		"id":     "",
		"tags":   []interface{}{},
		"active": "",
	}

	// We always validate metadata before passing to this function
	// Any unexpected format is a service level
	if metadata, ok := newMetadata[MetadataLabel]; ok {
		if dataMap, ok := metadata.(map[string]interface{}); ok {
			// Fan out metadata to SNS
			d.fanoutToSNS(emptyMetaData, dataMap, refresh)
		} else {
			log.Error("metadata field is not expected as a map")
		}
	} else {
		return
	}
}

// fanoutToSNS
// oldMetadata and new Metadata must be validated
func (d *DataProcessor) fanoutToSNS(oldMetadata, newMetadata map[string]interface{}, refresh bool) {
	metadataDelta := buildMetaDataDelta(oldMetadata, newMetadata)

	// if refresh is set to true set refresh key to true so down stream systems know this is the current state
	if refresh {
		metadataDelta["refresh"] = true
	}

	gameData := d.buildGameData(metadataDelta)

	if len(gameData.MetadataDelta) == 0 {
		return
	}

	if err := d.snsPublisher.Publish(gameData); err != nil {
		log.Error(err)
	}

	log.Trace(fmt.Sprintf("Game data published to metadata sns %v", gameData))
}

// buildMetaDataDelta returns differences in 2 states of metadata
// It returns a map if changes for each field that is different bwtween the older state and the newer state
// For strings it returns a key with that new value
// For string arrays it returns a map of string arrays with the keys `add` and `remove`
func buildMetaDataDelta(oldState, newState map[string]interface{}) map[string]interface{} {

	changes := make(map[string]interface{})

	for key, value := range newState {
		if _, isList := value.([]interface{}); isList {
			// TODO: what if old state doesnt have this key (as nil)?
			oldStateStringList := convertInterfaceToStringList(oldState[key].([]interface{}))
			newStateStringList := convertInterfaceToStringList(value.([]interface{}))
			toAdd := difference(newStateStringList, oldStateStringList)
			toRemove := difference(oldStateStringList, newStateStringList)
			if len(toAdd) > 0 || len(toRemove) > 0 {
				changes[key] = map[string]interface{}{
					"add":    toAdd,
					"remove": toRemove,
				}
			}
		} else {
			if value != oldState[key] {
				changes[key] = value
			}
		}
	}

	return changes
}

func (d *DataProcessor) buildGameData(metadataDelta map[string]interface{}) models.GameData {
	gameData := models.GameData{
		BroadcasterIDs: d.connectionMetaData.BroadcasterIDs,
		Env:            d.connectionMetaData.Env,
		GameID:         d.connectionMetaData.GameID,
		MetadataDelta:  metadataDelta,
	}

	return gameData
}

// difference returns the an array of strings from the first array of strings which are missing from the second
// array of strings
func difference(a, b []string) (diff []string) {
	m := make(map[string]bool)

	for _, item := range b {
		m[item] = true
	}

	for _, item := range a {
		if _, ok := m[item]; !ok {
			diff = append(diff, item)
		}

	}
	return
}

// convertInterfaceToStringList converts and array of interfaces
func convertInterfaceToStringList(t []interface{}) []string {
	s := make([]string, len(t))
	for i, v := range t {
		s[i] = fmt.Sprint(v)
	}

	return s
}
