package processor

import (
	"code.justin.tv/devhub/twitch-e2-ingest/models"
)

func (suite *ProcessorTest) TestFanoutToSNS() {
	oldMetaData := map[string]interface{}{
		"active": false,
		"tags": []interface{}{
			"xyz",
		},
		"id": "TestID1",
	}

	newMetaDataAllFields := map[string]interface{}{
		"active": true,
		"tags": []interface{}{
			"abcd",
			"1234",
		},
		"id": "TestID2",
	}

	gameData := models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"id":     "TestID2",
			"tags": map[string]interface{}{
				"add":    []string{"abcd", "1234"},
				"remove": []string{"xyz"},
			},
		},
	}

	suite.snsPublisher.On("Publish", gameData).Return(nil).Once()

	suite.processor.fanoutToSNS(oldMetaData, newMetaDataAllFields, false)
}

func (suite *ProcessorTest) TestFanoutToSNSRefresh() {
	oldMetaData := map[string]interface{}{
		"active": false,
		"tags": []interface{}{
			"xyz",
		},
		"id": "TestID1",
	}

	newMetaDataAllFields := map[string]interface{}{
		"active": true,
		"tags": []interface{}{
			"abcd",
			"1234",
		},
		"id": "TestID2",
	}

	gameData := models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"id":     "TestID2",
			"tags": map[string]interface{}{
				"add":    []string{"abcd", "1234"},
				"remove": []string{"xyz"},
			},
			"refresh": true,
		},
	}

	suite.snsPublisher.On("Publish", gameData).Return(nil).Once()

	suite.processor.fanoutToSNS(oldMetaData, newMetaDataAllFields, true)
}

func (suite *ProcessorTest) TestBuildMetaDataDelta() {
	oldMetaData := map[string]interface{}{
		"active": false,
		"tags": []interface{}{
			"xyz",
		},
		"id": "TestID1",
	}

	newMetaDataAllFields := map[string]interface{}{
		"active": true,
		"tags": []interface{}{
			"abcd",
			"1234",
		},
		"id": "TestID2",
	}

	newMetaDataString := map[string]interface{}{
		"string": "text",
	}

	newMetaDataBoolean := map[string]interface{}{
		"boolean": true,
	}

	newMetaDataNumber := map[string]interface{}{
		"number": 1,
	}

	newMetaDataInterfaceList := map[string]interface{}{
		"strings": []interface{}{
			"abcd",
			"1234",
		},
	}

	// Success case for all metadata fields
	suite.EqualValues(map[string]interface{}{
		"active": true,
		"tags": map[string]interface{}{
			"add": []string{
				"abcd",
				"1234",
			},
			"remove": []string{
				"xyz",
			},
		},
		"id": "TestID2",
	}, buildMetaDataDelta(oldMetaData, newMetaDataAllFields))

	// should detect when string feild changes
	suite.EqualValues(map[string]interface{}{
		"string": "text",
	}, buildMetaDataDelta(map[string]interface{}{}, newMetaDataString))

	// should detect when an number field changes
	suite.EqualValues(map[string]interface{}{
		"number": 1,
	}, buildMetaDataDelta(map[string]interface{}{}, newMetaDataNumber))

	// should detect when a boolean field changes
	suite.EqualValues(map[string]interface{}{
		"boolean": true,
	}, buildMetaDataDelta(map[string]interface{}{}, newMetaDataBoolean))

	// should detect when a interface list changes and return key with a struct with added and removed values
	suite.EqualValues(map[string]interface{}{
		"strings": map[string]interface{}{
			"add": []string{
				"abcd",
				"1234",
			},
			"remove": []string(nil),
		},
	}, buildMetaDataDelta(map[string]interface{}{"strings": []interface{}{}}, newMetaDataInterfaceList))

	// There are no differences between the old data and the new data
	suite.EqualValues(map[string]interface{}{}, buildMetaDataDelta(oldMetaData, oldMetaData))
}

func (suite *ProcessorTest) TestBuildGameData() {
	// Success case
	suite.EqualValues(models.GameData{
		BroadcasterIDs: suite.processor.connectionMetaData.BroadcasterIDs,
		GameID:         suite.processor.connectionMetaData.GameID,
		Env:            suite.processor.connectionMetaData.Env,
		MetadataDelta:  map[string]interface{}{},
	}, suite.processor.buildGameData(map[string]interface{}{}))
}

func (suite *ProcessorTest) TestDifference() {
	stringListA := []string{
		"abcd",
		"1234",
	}

	stringListSameAsA := []string{
		"abcd",
		"1234",
	}

	stringListDifferent := []string{
		"test",
		"anotherthing",
		"abcd",
		"1234",
	}

	// 2 string lists with equal values return an empty array
	suite.EqualValues([]string(nil), difference(stringListA, stringListSameAsA))

	// 2 string lists with unequal values returns values missing from first array
	suite.EqualValues([]string{"test", "anotherthing"}, difference(stringListDifferent, stringListA))

	// Returns nil string array if 1st array is a subset of the second array
	suite.EqualValues([]string(nil), difference(stringListA, stringListDifferent))
}

func (suite *ProcessorTest) TestConvertInterfaceToStringList() {
	interfaceList := []interface{}{
		"abcd",
		"1234",
	}

	expectedStringList := []string{
		"abcd",
		"1234",
	}

	interfaceListWithRandomEntrys := []interface{}{
		"abcd",
		2,
		map[string]string{},
	}

	expectedStringListFromRandomEntrys := []string{
		"abcd",
		"2",
		"map[]",
	}

	// returns string array from arary of interfaces
	suite.EqualValues(expectedStringList, convertInterfaceToStringList(interfaceList))
	suite.EqualValues(expectedStringListFromRandomEntrys, convertInterfaceToStringList(interfaceListWithRandomEntrys))

	// returns empty string array if there are no interfaces in list
	suite.EqualValues([]string{}, convertInterfaceToStringList([]interface{}{}))
}
