package processor

import (
	"encoding/json"
	"errors"
	"time"

	models "code.justin.tv/devhub/mdaas-ingest/models"
	. "code.justin.tv/devhub/mdaas-ingest/test_utils"
	e2models "code.justin.tv/devhub/twitch-e2-ingest/models"
)

func (suite *ProcessorTest) TestProcessCloseSignal() {
	// For delta, seqStart remains the same
	suite.processor.deltaPublishCount = 1
	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	data := &models.DeltaMsg{}
	err := json.Unmarshal([]byte(StandardDeltaPackWithDeltaOps), data)
	suite.NoError(err)

	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
			"refresh": true,
		},
	}).Return(nil).Once()
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		Active:         false,
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	resp := suite.processor.ProcessCloseSignal(models.MessageInfo{Message: "close_channel_signal", Time: timeStamp})
	suite.Nil(resp)

	time.Sleep(10 * time.Millisecond)

	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		Active:         false,
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestProcessCloseSignal_SNSPublishError() {
	// For delta, seqStart remains the same
	suite.processor.deltaPublishCount = 1
	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	data := &models.DeltaMsg{}
	err := json.Unmarshal([]byte(StandardDeltaPackWithDeltaOps), data)
	suite.NoError(err)

	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
			"refresh": true,
		},
	}).Return(nil).Once()
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		Active:         false,
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(errors.New("Failed")).Once()

	resp := suite.processor.ProcessCloseSignal(models.MessageInfo{Message: "close_channel_signal", Time: timeStamp})
	suite.Error(resp)

	time.Sleep(10 * time.Millisecond)

	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		Active:         false,
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID,
		SessionID:      TestSessionID,
	})
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestProcessCloseSignal_NoMetadata() {

	suite.processor.connectionMetaData.ClientID = ""
	suite.connectionData.ClientID = ""

	// For delta, seqStart remains the same
	suite.processor.deltaPublishCount = 1
	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	data := &models.DeltaMsg{}
	err := json.Unmarshal([]byte(StandardDeltaPackWithDeltaOps), data)
	suite.NoError(err)

	resp := suite.processor.ProcessCloseSignal(models.MessageInfo{Message: "close_channel_signal", Time: timeStamp})
	suite.Nil(resp)

	time.Sleep(10 * time.Millisecond)

	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		Active:         false,
		ClientID:       "",
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID,
		SessionID:      TestSessionID,
	})
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}
