package processor

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/devhub/mdaas-ingest/internal/metrics"
	log "github.com/sirupsen/logrus"

	models "code.justin.tv/devhub/mdaas-ingest/models"
	e2models "code.justin.tv/devhub/twitch-e2-ingest/models"
)

func (d *DataProcessor) processConnectPack(connectPack models.ConnectMsg, timestamp int64) interface{} {
	d.pauseOnErr = false

	if d.connectionMetaData.ClientID != "" {
		return &e2models.ConnectReAuthNotSupported
	}

	// authorize connection
	if authErr := d.authorizeConnection(connectPack.Body.Token, connectPack.Body.GameID, connectPack.Body.BroadcasterIDs); authErr != nil {
		d.connectionMetaData = models.ConnectionMetaData{}
		return authErr
	}

	// Check if required game data and metadata all exist
	if err := d.validateConnectPackRequiredInfo(connectPack); err != nil {
		d.connectionMetaData = models.ConnectionMetaData{}
		return err
	}

	// Set data in memory
	d.data = connectPack.Body.Data
	d.connectionMetaData.GameID = connectPack.Body.GameID
	// Set connection id to where the first correct connect message was received
	d.connectionMetaData.ConnectionID = timestamp
	d.connectionMetaData.Env = connectPack.Body.Env
	d.connectionMetaData.SessionID = connectPack.Body.SessionID
	if len(d.connectionMetaData.BroadcasterIDs) == 0 {
		d.connectionMetaData.BroadcasterIDs = connectPack.Body.BroadcasterIDs
	}

	// Force active tag to true when initial connection
	d.connectionMetaData.Active = true
	d.debug = connectPack.Body.Debug

	// When initial connect, set up the map for reference of broadcaster id during kinesis publishes
	// There is no need to set first key frame map if we already set this one
	cloneNum := d.publisher.GetCloneNumByGame(d.connectionMetaData.GameID)
	d.cloneToBroadcasterInfo = make([][]models.BroadcasterKinesisInfo, cloneNum)
	for clone := range d.cloneToBroadcasterInfo {
		infos := make([]models.BroadcasterKinesisInfo, len(d.connectionMetaData.BroadcasterIDs))
		for i, id := range d.connectionMetaData.BroadcasterIDs {
			info := models.BroadcasterKinesisInfo{
				BroadcasterID: id,
				FirstKeyFrame: "",
				LastFullState: "",
			}
			infos[i] = info
		}
		d.cloneToBroadcasterInfo[clone] = infos
	}

	// Fan out metadata to SNS
	go d.fanoutFullToSNS(d.data, false)
	// Save valid debug log into S3
	connectDataWithMaskedToken, jsonErr := maskTokenInConnectMessage(*connectPack.Body)
	if jsonErr == nil {
		go d.saveValidReplayDataInS3(string(connectDataWithMaskedToken))
	} else {
		log.Errorf("Caught JSON error when parsing string with masked token: %v\n", jsonErr)
	}

	// Fan out to kinesis
	errResp := d.fanoutToFullStateKinesis()
	if errResp != nil {
		return errResp
	}

	// Fanout SNS full data to SNS
	errResp = d.fanoutSNSFullData(timestamp)
	if errResp != nil {
		return errResp
	}

	// Report connect message process time
	metrics.Reporter().Report("ProcessTimeConnect", metrics.GetDurationByMS(timestamp), telemetry.UnitSeconds)

	return &models.ConnectedMsg{Connected: true}
}
