package processor

import (
	"encoding/json"
	"time"

	"code.justin.tv/devhub/mdaas-ingest/models"
	e2models "code.justin.tv/devhub/twitch-e2-ingest/models"

	. "code.justin.tv/devhub/mdaas-ingest/test_utils"

	"github.com/stretchr/testify/mock"
)

func (suite *ProcessorTest) TestProcessValidConnectPackWithUserToken() {
	// reset processor to initial state for connect pack
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)

	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	owlResp := e2models.AuthResp{ClientID: TestClientID, UserID: TestBroadcasterID}
	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)

	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: []string{TestBroadcasterID},
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
		},
	}).Return(nil).Once()

	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		Active:         true,
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: []string{TestBroadcasterID},
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   timeStamp,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.UserToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	suite.publisher.On("GetCloneNumByGame", TestGameID).Return(2)
	suite.publisher.On("Publish", mock.Anything, models.Event{Full: &models.FullEvent{
		ClientID:     suite.connectionData.ClientID,
		Env:          suite.connectionData.Env,
		GameID:       suite.connectionData.GameID,
		Data:         suite.data,
		IsServerData: false,
	}}, GetCloneToBroadcasterMapping(UserTokenFirstMessage)).Return(suite.cloneToKinesisInfo, nil)

	resp := suite.processor.ProcessDataPack(models.MessageInfo{Message: StandardConnectionDataUserToken, Time: timeStamp})
	time.Sleep(10 * time.Millisecond)

	result, err := json.Marshal(resp)
	suite.Nil(err)
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
	suite.Equal(string(result), "{\"connected\":true}")
	suite.Equal(suite.cloneToKinesisInfo, suite.processor.cloneToBroadcasterInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		Active:         true,
		ClientID:       TestClientID,
		BroadcasterIDs: []string{TestBroadcasterID},
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   false,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
}

func (suite *ProcessorTest) TestProcessValidConnectPackWithAppToken() {
	// reset processor to initial state for connect pack
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)
	owlResp := e2models.AuthResp{ClientID: TestClientID}

	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)
	suite.auth.On("IfTrustedSource", mock.Anything, suite.connectionData.BroadcasterIDs, owlResp).Return(true)

	suite.publisher.On("GetCloneNumByGame", TestGameID).Return(2)
	suite.publisher.On("Publish", mock.Anything, suite.fullEvent, suite.initialMapping).Return(suite.cloneToKinesisInfo, nil)
	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
		},
	}).Return(nil).Once()

	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		Active:         true,
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   timeStamp,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardConnectionDataMultiBroadcaster, timeStamp))

	time.Sleep(10 * time.Millisecond)

	result, err := json.Marshal(resp)
	suite.Nil(err)
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
	suite.Equal(string(result), "{\"connected\":true}")
	suite.Equal(suite.cloneToKinesisInfo, suite.processor.cloneToBroadcasterInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		Active:         true,
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      suite.connectionData.SessionID,
	})
}

// Test connect pack path - with debug flag
func (suite *ProcessorTest) TestProcessValidConnectPackWithUserToken_DebugOn() {
	// reset processor to initial state for connect pack
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)

	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	owlResp := e2models.AuthResp{ClientID: TestClientID, UserID: TestBroadcasterID}
	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)

	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: []string{TestBroadcasterID},
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
		},
	}).Return(nil).Once()

	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		Active:         true,
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: []string{TestBroadcasterID},
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   timeStamp,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.UserToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	suite.publisher.On("GetCloneNumByGame", TestGameID).Return(2)
	suite.publisher.On("Publish", mock.Anything, models.Event{Full: &models.FullEvent{
		ClientID:     suite.connectionData.ClientID,
		Env:          suite.connectionData.Env,
		GameID:       suite.connectionData.GameID,
		Data:         suite.data,
		IsServerData: false,
	}}, GetCloneToBroadcasterMapping(UserTokenFirstMessage)).Return(suite.cloneToKinesisInfo, nil)

	testMessageInfo, err := SetDebugFlagOnConnectPackMsg(GenerateMessageInfoFromRawMessage(StandardConnectionDataUserToken, timeStamp), true)
	suite.NoError(err)
	maskedRawMessage, jsonErr := MaskTokenInMessageInfo(*testMessageInfo)
	suite.NoError(jsonErr)
	expectedDebugLogJSON, jsonErr := GenerateDebugLogJSON(suite.connectionData.ClientID, suite.connectionData.GameID, []string{TestBroadcasterID}, *maskedRawMessage)
	suite.NoError(jsonErr)
	suite.publisher.On("SavePublishedStateDataInS3", expectedDebugLogJSON).Return()
	resp := suite.processor.ProcessDataPack(*testMessageInfo)

	time.Sleep(10 * time.Millisecond)

	result, err := json.Marshal(resp)
	suite.Nil(err)
	suite.Equal(string(result), "{\"connected\":true}")
	suite.Equal(suite.cloneToKinesisInfo, suite.processor.cloneToBroadcasterInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		Active:         true,
		ClientID:       TestClientID,
		BroadcasterIDs: []string{TestBroadcasterID},
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   false,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
}

func (suite *ProcessorTest) TestProcessValidConnectPackWithAppToken_DebugOn() {
	// reset processor to initial state for connect pack
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)
	owlResp := e2models.AuthResp{ClientID: TestClientID}

	originalMessageID := suite.processor.connectionMetaData.MessageID
	timeStamp := time.Now().UnixNano()

	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)
	suite.auth.On("IfTrustedSource", mock.Anything, suite.connectionData.BroadcasterIDs, owlResp).Return(true)

	suite.publisher.On("GetCloneNumByGame", TestGameID).Return(2)
	suite.publisher.On("Publish", mock.Anything, suite.fullEvent, suite.initialMapping).Return(suite.cloneToKinesisInfo, nil)
	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
		},
	}).Return(nil).Once()

	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		Active:         true,
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   timeStamp,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	testMessageInfo, err := SetDebugFlagOnConnectPackMsg(GenerateMessageInfoFromRawMessage(StandardConnectionDataMultiBroadcaster, timeStamp), true)
	suite.NoError(err)
	maskedRawMessage, jsonErr := MaskTokenInMessageInfo(*testMessageInfo)
	suite.NoError(jsonErr)
	expectedDebugLogJSON, jsonErr := GenerateDebugLogJSON(suite.connectionData.ClientID, suite.connectionData.GameID, suite.connectionData.BroadcasterIDs, *maskedRawMessage)
	suite.NoError(jsonErr)
	suite.publisher.On("SavePublishedStateDataInS3", expectedDebugLogJSON).Return()
	resp := suite.processor.ProcessDataPack(*testMessageInfo)

	time.Sleep(10 * time.Millisecond)

	result, err := json.Marshal(resp)
	suite.Nil(err)
	suite.Equal(string(result), "{\"connected\":true}")
	suite.Equal(suite.cloneToKinesisInfo, suite.processor.cloneToBroadcasterInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		Active:         true,
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      suite.connectionData.SessionID,
	})
}

func (suite *ProcessorTest) TestProcessInvalidConnectPack_MissingID() {
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)

	owlResp := e2models.AuthResp{ClientID: TestClientID, UserID: TestBroadcasterID}
	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)

	resp := suite.processor.ProcessDataPack(models.MessageInfo{Message: ConnectionBadMetadataMissingID, Time: time.Now().UnixNano()})
	expectedErr := e2models.BuildErrMsgByField(e2models.ConnectMissingInfo, e2models.SessionIDErr)
	suite.Equal(resp, expectedErr)
}

func (suite *ProcessorTest) TestProcessInvalidConnectPack_MissingID_DebugOn() {
	suite.processor.debug = true
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)

	owlResp := e2models.AuthResp{ClientID: TestClientID, UserID: TestBroadcasterID}
	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)

	resp := suite.processor.ProcessDataPack(models.MessageInfo{Message: ConnectionBadMetadataMissingID, Time: time.Now().UnixNano()})
	expectedErr := e2models.BuildErrMsgByField(e2models.ConnectMissingInfo, e2models.SessionIDErr)
	suite.Equal(resp, expectedErr)
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestProcessInvalidConnectPack_BadTag() {
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)

	owlResp := e2models.AuthResp{ClientID: TestClientID, UserID: TestBroadcasterID}
	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)

	resp := suite.processor.ProcessDataPack(models.MessageInfo{Message: ConnectionBadTags, Time: time.Now().UnixNano()})
	expectedErr := e2models.BuildErrMsgByField(e2models.ConnectBadInfo, e2models.TagErr)
	suite.Equal(resp, expectedErr)
}

func (suite *ProcessorTest) TestProcessInvalidConnectPack_BadTag_DebugOn() {
	suite.processor.debug = true
	suite.processor = NewDataProcessor(suite.auth, suite.publisher, suite.snsPublisher, suite.buffer)

	owlResp := e2models.AuthResp{ClientID: TestClientID, UserID: TestBroadcasterID}
	suite.auth.On("ValidateClient", mock.Anything, TestToken).Return(owlResp, nil)
	suite.auth.On("IfWhitelistedClient", mock.Anything, owlResp, TestGameID).Return(nil)

	resp := suite.processor.ProcessDataPack(models.MessageInfo{Message: ConnectionBadTags, Time: time.Now().UnixNano()})
	expectedErr := e2models.BuildErrMsgByField(e2models.ConnectBadInfo, e2models.TagErr)
	suite.Equal(resp, expectedErr)
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}
