package processor

import (
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	jdelta "code.justin.tv/devhub/json-delta/lib"
	"code.justin.tv/devhub/mdaas-ingest/internal/metrics"
	models "code.justin.tv/devhub/mdaas-ingest/models"
	e2models "code.justin.tv/devhub/twitch-e2-ingest/models"

	log "github.com/sirupsen/logrus"
)

func (d *DataProcessor) processDeltaPack(deltaPack models.DeltaMsg, messageInfo models.MessageInfo) *e2models.ErrorMsg {
	if d.pauseOnErr && d.connectionMetaData.ClientID != "" {
		return &e2models.WaitingOnRefreshPack
	}

	// Check if the connection is authed first
	if d.connectionMetaData.ClientID == "" {
		return &e2models.ConnectionNotAuthed
	}

	oldState := make(map[string]interface{})
	if metadata, ok := d.data[MetadataLabel].(map[string]interface{}); ok {
		deepCopyMetadata(oldState, metadata)
	}

	// Merge delta pack into full state
	updated, err := d.patchDeltas(d.data, deltaPack.Data)
	if err != nil {
		return &e2models.DeltaInvalid
	}

	// Validate if metadata field is still valid
	if validatedErr := validateMetadata(updated); validatedErr != nil {
		return e2models.BuildErrMsgByField(e2models.DeltaMetadataFieldImpaired, validatedErr.Field)
	}

	// save debug logs for CME team into S3
	go d.saveValidReplayDataInS3(messageInfo.Message)

	if metadata, ok := updated[MetadataLabel]; ok {
		if dataMap, ok := metadata.(map[string]interface{}); ok {
			// Fan out metadata delta to SNS
			go d.fanoutToSNS(oldState, dataMap, false)
		}
	}

	d.data = updated

	// Fan out to kinesis
	errResp := d.fanoutToDeltaKinesis(deltaPack.Data)
	if errResp != nil {
		return errResp
	}

	// Fan out full data to SNS
	errResp = d.fanoutSNSFullData(messageInfo.Time)
	if errResp != nil {
		return errResp
	}

	// Report delta message process time
	metrics.Reporter().Report("ProcessTimeDelta", metrics.GetDurationByMS(messageInfo.Time), telemetry.UnitSeconds)
	return nil
}

// deepCopyMetadata copies metadataReference to metadataCopy
func deepCopyMetadata(metadataCopy, metadataReference map[string]interface{}) {
	for key, value := range metadataReference {
		if list, isList := value.([]interface{}); isList {
			listCopy := make([]interface{}, len(list))
			copy(listCopy, list)
			metadataCopy[key] = listCopy
		} else {
			metadataCopy[key] = value
		}
	}
}

func (d *DataProcessor) patchDeltas(fullState map[string]interface{}, deltaStr []interface{}) (state map[string]interface{}, err error) {
	defer func() {
		if r := recover(); r != nil {
			err = r.(error)
			log.Errorf("Client %s game %s sent a bad delta message: %s. Error Message %v\n ", d.GetConnectionClientID(), d.connectionMetaData.GameID, deltaStr, err)
		}
	}()

	return jdelta.PatchDeltas(fullState, deltaStr)
}
