package processor

import (
	"encoding/json"
	"errors"
	"reflect"
	"time"

	jdelta "code.justin.tv/devhub/json-delta/lib"
	models "code.justin.tv/devhub/mdaas-ingest/models"
	. "code.justin.tv/devhub/mdaas-ingest/test_utils"
	e2models "code.justin.tv/devhub/twitch-e2-ingest/models"
	"github.com/stretchr/testify/mock"
)

func (suite *ProcessorTest) TestProcessValidDeltaPack() {
	// For delta, seqStart remains the same
	suite.processor.deltaPublishCount = 1
	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	data := &models.DeltaMsg{}
	err := json.Unmarshal([]byte(StandardDeltaPackWithDeltaOps), data)
	suite.NoError(err)

	delta := models.DeltaEvent{
		GameID: suite.connectionData.GameID,
		Data:   data.Data,
	}

	suite.publisher.On("Publish", mock.Anything, models.Event{Delta: &delta}, suite.cloneToKinesisInfo).Return(suite.cloneToKinesisInfo, nil)
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardDeltaPackWithDeltaOps, timeStamp))
	suite.Nil(resp)
	suite.Equal(suite.processor.deltaPublishCount, 2)
	suite.Equal(suite.processor.cloneToBroadcasterInfo, suite.cloneToKinesisInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestProcessValidDeltaPackWithMetadataChange() {
	// For delta, seqStart remains the same
	suite.processor.deltaPublishCount = 1
	originalMessageID := suite.processor.connectionMetaData.MessageID

	data := &models.DeltaMsg{}
	err := json.Unmarshal([]byte(StandardDeltaPackChangeMetadata), data)
	suite.NoError(err)

	delta := &models.DeltaEvent{
		GameID: suite.connectionData.GameID,
		Data:   data.Data,
	}

	suite.publisher.On("Publish", mock.Anything, models.Event{Delta: delta}, suite.cloneToKinesisInfo).Return(suite.cloneToKinesisInfo, nil)
	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"tags": map[string]interface{}{
				"add":    []string{TestTag3},
				"remove": []string{TestTag1, TestTag2},
			},
		},
	}).Return(nil).Once()

	timeStamp := time.Now().UnixNano()
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardDeltaPackChangeMetadata, timeStamp))

	time.Sleep(10 * time.Millisecond)

	suite.Nil(resp)
	suite.Equal(suite.processor.deltaPublishCount, 2)
	suite.Equal(suite.processor.cloneToBroadcasterInfo, suite.cloneToKinesisInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestProcessInvalidDeltaMessage_NotAuthed() {
	suite.processor.connectionMetaData.ClientID = ""
	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardDeltaPackChangeMetadata, time.Now().UnixNano()))
	suite.Equal(resp, &e2models.ConnectionNotAuthed)
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestDeepCopyMetadata() {
	stateCopy := make(map[string]interface{})
	stateEmpty := make(map[string]interface{})
	stateNormal := map[string]interface{}{
		"active": true,
		"tags": []interface{}{
			"abcd",
			"1234",
		},
		"id": "TestID2",
	}

	// creates a deep copy with metadata attributes
	deepCopyMetadata(stateCopy, stateNormal)
	suite.Equal(stateCopy, stateNormal)
	suite.NotEqual(reflect.ValueOf(stateCopy["tags"]).Pointer(), reflect.ValueOf(stateNormal["tags"]).Pointer())

	// still creates a copy with empty map
	stateCopy = make(map[string]interface{})
	deepCopyMetadata(stateCopy, stateEmpty)
	suite.Equal(stateCopy, stateEmpty)
}

// test with debug flag on
func (suite *ProcessorTest) TestProcessValidDeltaPack_DebugOn() {
	// For delta, seqStart remains the same
	suite.processor.debug = true
	suite.processor.deltaPublishCount = 1
	timeStamp := time.Now().UnixNano()
	originalMessageID := suite.processor.connectionMetaData.MessageID

	data := &models.DeltaMsg{}
	err := json.Unmarshal([]byte(StandardDeltaPackWithDeltaOps), data)
	suite.NoError(err)

	delta := models.DeltaEvent{
		GameID: suite.connectionData.GameID,
		Data:   data.Data,
	}

	suite.publisher.On("Publish", mock.Anything, models.Event{Delta: &delta}, suite.cloneToKinesisInfo).Return(suite.cloneToKinesisInfo, nil)
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	expectedDebugLogJSON, err := GenerateDebugLogJSON(TestClientID, TestGameID, suite.connectionData.BroadcasterIDs, StandardDeltaPackWithDeltaOps)
	suite.NoError(err)
	suite.publisher.On("SavePublishedStateDataInS3", expectedDebugLogJSON).Return()
	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardDeltaPackWithDeltaOps, timeStamp))

	time.Sleep(10 * time.Millisecond)
	suite.Nil(resp)
	suite.Equal(suite.processor.deltaPublishCount, 2)
	suite.Equal(suite.processor.cloneToBroadcasterInfo, suite.cloneToKinesisInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
}

// test with debug flag on
func (suite *ProcessorTest) TestProcessValidDeltaPackWithMetadataChange_DebugOn() {
	// For delta, seqStart remains the same
	suite.processor.debug = true
	suite.processor.deltaPublishCount = 1
	originalMessageID := suite.processor.connectionMetaData.MessageID

	data := &models.DeltaMsg{}
	err := json.Unmarshal([]byte(StandardDeltaPackChangeMetadata), data)
	suite.NoError(err)

	delta := &models.DeltaEvent{
		GameID: suite.connectionData.GameID,
		Data:   data.Data,
	}

	suite.publisher.On("Publish", mock.Anything, models.Event{Delta: delta}, suite.cloneToKinesisInfo).Return(suite.cloneToKinesisInfo, nil)
	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"tags": map[string]interface{}{
				"add":    []string{TestTag3},
				"remove": []string{TestTag1, TestTag2},
			},
		},
	}).Return(nil).Once()

	timeStamp := time.Now().UnixNano()
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	expectedDebugLogJSON, err := GenerateDebugLogJSON(TestClientID, TestGameID, suite.connectionData.BroadcasterIDs, StandardDeltaPackChangeMetadata)
	suite.NoError(err)
	suite.publisher.On("SavePublishedStateDataInS3", expectedDebugLogJSON).Return()
	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardDeltaPackChangeMetadata, timeStamp))

	time.Sleep(10 * time.Millisecond)

	suite.Nil(resp)
	suite.Equal(suite.processor.deltaPublishCount, 2)
	suite.Equal(suite.processor.cloneToBroadcasterInfo, suite.cloneToKinesisInfo)
	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
}

func (suite *ProcessorTest) Test_PatchDeltas() {
	state := map[string]interface{}{
		"key1": "a",
	}

	deltaStr := `[["key1", "b"]]`
	deltas, err := jdelta.DeltaStateStringToInterface(deltaStr)
	suite.NoError(err)

	fullstate, err := suite.processor.patchDeltas(state, deltas)
	suite.NoError(err)

	expected := map[string]interface{}{
		"key1": "b",
	}

	suite.Equal(expected, fullstate)
}

func (suite *ProcessorTest) Test_PatchDeltasInvalidPath() {
	state := map[string]interface{}{
		"key1": "a",
	}

	deltaStr := `[["", "b"]]`
	deltas, err := jdelta.DeltaStateStringToInterface(deltaStr)
	suite.NoError(err)

	fullstate, err := suite.processor.patchDeltas(state, deltas)
	suite.Error(err)

	expected := map[string]interface{}(nil)

	suite.Equal(expected, fullstate)
}

func (suite *ProcessorTest) Test_PatchDeltasEmptyPath() {
	state := map[string]interface{}{
		"key1": "a",
	}

	deltaStr := `[[]]`
	deltas, err := jdelta.DeltaStateStringToInterface(deltaStr)
	suite.NoError(err)

	fullstate, err := suite.processor.patchDeltas(state, deltas)
	suite.Error(err)
	suite.Equal(err, errors.New("empty delta message"))

	expected := map[string]interface{}(nil)

	suite.Equal(expected, fullstate)
}

func (suite *ProcessorTest) Test_PatchDeltasConsecutivePeriods() {
	state := map[string]interface{}{
		"key1": "a",
	}

	deltaStr := `[["key1..a", "b"]]`
	deltas, err := jdelta.DeltaStateStringToInterface(deltaStr)
	suite.NoError(err)

	fullstate, err := suite.processor.patchDeltas(state, deltas)
	suite.Error(err)
	suite.Equal(err, errors.New("invalid delta format"))

	expected := map[string]interface{}(nil)

	suite.Equal(expected, fullstate)
}

func (suite *ProcessorTest) Test_PatchDeltasEmptyBrackets() {
	state := map[string]interface{}{
		"key1": "a",
	}

	deltaStr := `[["key1.a[].test", "b"]]`
	deltas, err := jdelta.DeltaStateStringToInterface(deltaStr)
	suite.NoError(err)

	fullstate, err := suite.processor.patchDeltas(state, deltas)
	suite.Error(err)
	suite.Equal(err, errors.New("invalid delta format"))

	expected := map[string]interface{}(nil)

	suite.Equal(expected, fullstate)
}
