package processor

import (
	"time"

	models "code.justin.tv/devhub/mdaas-ingest/models"
	. "code.justin.tv/devhub/mdaas-ingest/test_utils"
	e2models "code.justin.tv/devhub/twitch-e2-ingest/models"
	"github.com/stretchr/testify/mock"
)

func (suite *ProcessorTest) TestProcessValidRefreshPack() {
	originalMessageID := suite.processor.connectionMetaData.MessageID
	updated := GetCloneToBroadcasterMapping(UpdatedMessage)
	suite.publisher.On("Publish", mock.Anything, suite.fullEvent, suite.cloneToKinesisInfo).Return(updated, nil)
	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
			"refresh": true,
		},
	}).Return(nil).Once()

	timeStamp := time.Now().UnixNano()
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardRefresh, timeStamp))

	time.Sleep(10 * time.Millisecond)

	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
	suite.Nil(resp)
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

// when debug flag is on the log should be triggered
func (suite *ProcessorTest) TestProcessValidRefreshPackWithDebugFlagOn() {
	suite.processor.debug = true
	originalMessageID := suite.processor.connectionMetaData.MessageID
	updated := GetCloneToBroadcasterMapping(UpdatedMessage)
	suite.publisher.On("Publish", mock.Anything, suite.fullEvent, suite.cloneToKinesisInfo).Return(updated, nil)
	suite.snsPublisher.On("Publish", e2models.GameData{
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		Env:            suite.connectionData.Env,
		GameID:         suite.connectionData.GameID,
		MetadataDelta: map[string]interface{}{
			"active": true,
			"tags": map[string]interface{}{
				"add":    []string{TestTag1, TestTag2},
				"remove": []string(nil),
			},
			"refresh": true,
		},
	}).Return(nil).Once()

	timeStamp := time.Now().UnixNano()
	suite.snsPublisher.On("PublishByARN", e2models.GameFullData{
		ClientID:       suite.connectionData.ClientID,
		Env:            suite.connectionData.Env,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         suite.connectionData.GameID,
		Time:           timeStamp,
		ConnectionID:   suite.connectionData.ConnectionID,
		MessageID:      originalMessageID + 1,
		Data:           suite.data,
		SessionID:      suite.connectionData.SessionID,
		ClientType:     e2models.AppToken,
	}, "arn:aws:sns:us-west-2:797743463538:game-testGameID_client-testClientID_env-dev").Return(nil).Once()

	expectedDebugLog, jsonErr := GenerateDebugLogJSON(suite.connectionData.ClientID, suite.connectionData.GameID, suite.connectionData.BroadcasterIDs, StandardRefresh)
	suite.NoError(jsonErr)
	suite.publisher.On("SavePublishedStateDataInS3", expectedDebugLog).Return()

	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardRefresh, timeStamp))

	time.Sleep(10 * time.Millisecond)

	suite.Equal(suite.processor.connectionMetaData, models.ConnectionMetaData{
		ClientID:       TestClientID,
		BroadcasterIDs: suite.connectionData.BroadcasterIDs,
		GameID:         TestGameID,
		Env:            "dev",
		IsServerData:   true,
		ConnectionID:   suite.processor.connectionMetaData.ConnectionID,
		MessageID:      originalMessageID + 1,
		SessionID:      TestSessionID,
	})
	suite.Nil(resp)
}

func (suite *ProcessorTest) TestProcessValidRefreshPack_BadTags() {
	resp := suite.processor.ProcessDataPack(models.MessageInfo{Message: StandardRefreshBadTags, Time: time.Now().UnixNano()})
	suite.Equal(resp, e2models.BuildErrMsgByField(e2models.RefreshMissingInfo, e2models.TagErr))
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestProcessValidRefreshPack_BadTags_DebugOn() {
	suite.processor.debug = true
	resp := suite.processor.ProcessDataPack(models.MessageInfo{Message: StandardRefreshBadTags, Time: time.Now().UnixNano()})
	suite.Equal(resp, e2models.BuildErrMsgByField(e2models.RefreshMissingInfo, e2models.TagErr))
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}

func (suite *ProcessorTest) TestProcessRefreshMessage_NotAuthed() {
	suite.processor.connectionMetaData.ClientID = ""
	resp := suite.processor.ProcessDataPack(GenerateMessageInfoFromRawMessage(StandardRefresh, time.Now().UnixNano()))
	suite.Equal(resp, &e2models.ConnectionNotAuthed)
	suite.publisher.AssertNotCalled(suite.T(), "SavePublishedStateDataInS3")
}
