package ratelimiter

import (
	"fmt"
	"time"

	log "github.com/sirupsen/logrus"
)

//go:generate go run github.com/vektra/mockery/cmd/mockery -name RateLimiter
type RateLimiter interface {
	IfExceedLimit() bool
}

type rateLimiterImpl struct {
	count       int
	bucketStart time.Time
}

func NewRateLimiter() RateLimiter {
	return &rateLimiterImpl{0, getBucketStart(time.Now())}
}

func (r *rateLimiterImpl) IfExceedLimit() bool {
	log.Trace(fmt.Sprintf("Bucket size currently at %d", r.count))

	now := time.Now()
	bucketStart := getBucketStart(now)
	// Bucket starts at every starting 10s, e.g 0s, 10s, 20s... in a min
	if !bucketStart.Equal(r.bucketStart) {
		r.bucketStart = bucketStart
		r.count = 1
		return false
	}

	r.count = r.count + 1
	return r.count > 60
}

// rate-limit bucket is by 1 min (60 messages)
func getBucketStart(now time.Time) time.Time {
	return time.Date(now.Year(), now.Month(), now.Day(), now.Hour(), now.Minute(), 0, 0, now.Location())
}
