package server

import (
	"encoding/json"
	"math/rand"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/devhub/mdaas-ingest/internal/metrics"

	log "github.com/sirupsen/logrus"

	models "code.justin.tv/devhub/mdaas-ingest/models"
	e2SharedModels "code.justin.tv/devhub/twitch-e2-ingest/models"
)

func init() {
	rand.Seed(time.Now().UTC().UnixNano())
}

func (b *Binding) OnTextMessage(msg string) {
	if validationErr := b.validateMessage(msg); validationErr != nil {
		b.writeFeedback(*validationErr)
	}
	// Put data to buffer for sequential processing
	// Put message to channel
	b.processor.PutToBuffer(models.MessageInfo{Message: msg, Time: time.Now().UnixNano()})
}

func (b *Binding) sendReconnectPack(reconnectTime int) {
	// Always return a immediate reconnect pack for now
	// Should calculate a recommended reconnect time in the future
	reconnectPack := models.ReconnectMsg{Reconnect: reconnectTime}

	// Form reconnect response
	reconnectMessage, err := json.Marshal(reconnectPack)
	if err != nil {
		b.writeText(models.DefaultReconnectMsg)
		return
	}

	err = b.session.WriteBinaryAsText(reconnectMessage)
	if err != nil {
		log.Error(err)
	}
}

func (b *Binding) writeFeedback(resp interface{}) {
	feedback, err := json.Marshal(resp)
	// Marshall wil only fail if service is having expected downtime, disconnect
	if err != nil {
		b.session.Close(models.DefaultDisconnectErr)
		return
	}

	// If response is an error message
	if processErr, ok := (resp).(*e2SharedModels.ErrorMsg); ok {
		// Error reporting
		if processErr.ErrType == e2SharedModels.ServerErr {
			// Report unexpected error counts
			metrics.Reporter().Report("ServerFailures", 1.0, telemetry.UnitCount)
		} else if processErr.ErrType == e2SharedModels.BadInput {
			metrics.Reporter().Report("BadMessages", 1.0, telemetry.UnitCount)
		} else if processErr.ErrType == e2SharedModels.AuthErr {
			metrics.Reporter().Report("AuthFailures", 1.0, telemetry.UnitCount)
		} else if processErr.ErrType == e2SharedModels.BadRequestRate {
			metrics.Reporter().Report("BadRequestRate", 1.0, telemetry.UnitCount)
		}

		// Clean data upon any errors
		b.processor.CleanData()

		// Pause further delta processing
		b.processor.SetToPauseState()

		// write error back or request a reconnect or disconnect
		// Different actions based on error types
		switch processErr.Action {
		case e2SharedModels.ActionReconnect:
			b.write(feedback)
			b.sendReconnectPack(0)
			return
		case e2SharedModels.ActionDisconnect:
			b.write(feedback)
			// close buffer when server force closing the connection
			b.OnClosed(processErr)
			return
		case e2SharedModels.ActionDoNothing:
			return
		default:
		}
	}

	b.write(feedback)
}

func (b *Binding) write(feedback []byte) {
	err := b.session.WriteBinaryAsText(feedback)
	// Marshall wil only fail if service is having expected downtime or connection is already closed, disconnect
	if err != nil {
		b.session.Close(models.DefaultDisconnectErr)
		return
	}
}

func (b *Binding) writeText(feedback string) {
	err := b.session.WriteText(feedback)
	// Marshall wil only fail if service is having expected downtime or connection is already closed, disconnect
	if err != nil {
		b.session.Close(models.DefaultDisconnectErr)
		return
	}
}

func (b *Binding) WriteReconnectUponServiceClosure() {
	// Generate a delayed reconnect randomized from 0 - 5000ms
	delay := 1 + rand.Intn(4999)
	b.sendReconnectPack(delay)
}
