package server

import (
	"fmt"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/devhub/mdaas-ingest/internal/clients/kinesis"
	"code.justin.tv/devhub/mdaas-ingest/internal/metrics"
	"code.justin.tv/devhub/mdaas-ingest/models"
	"code.justin.tv/devhub/mdaas-ingest/processor"
	"code.justin.tv/devhub/mdaas-ingest/ratelimiter"
	"code.justin.tv/devhub/twitch-e2-ingest/authentication"
	"code.justin.tv/devhub/twitch-e2-ingest/sns"
	"code.justin.tv/karlpatr/message-prototype/libs/session"
)

const closeMessage = "close_channel_signal"

type server struct {
	kinesis    kinesis.Publisher
	sns        sns.Publisher
	authClient authentication.Auth
	bindings   *Bindings
}

func (s *server) Factory() session.BindingFactory { return s.create }

func NewServer(kinesis kinesis.Publisher, sns sns.Publisher, authClient authentication.Auth, bindings *Bindings) session.Server {
	return &server{kinesis, sns, authClient, bindings}
}

func (s *server) create(client session.Client) session.Binding {
	binding := newBinding(client, s.authClient, s.kinesis, s.sns, s.bindings)
	return binding
}

func newBinding(client session.Client, authClient authentication.Auth, kinesisPublisher kinesis.Publisher, snsPublisher sns.Publisher, bindings *Bindings) session.Binding {
	// initialize channel buffer
	workerQueue := make(chan models.MessageInfo, 300)

	// initialize processor
	processor := processor.NewDataProcessor(authClient, kinesisPublisher, snsPublisher, workerQueue)

	// create Binding
	binding := &Binding{client, processor, ratelimiter.NewRateLimiter(), false, bindings}

	// initialize metrics reporter
	go binding.reportMetrics()

	// assign work queue to dedicated worker
	go binding.handleDataStream(workerQueue)

	// Lock binding in memory to add
	defer func() {
		if bindings != nil {
			bindings.Unlock()
		}
	}()

	if bindings != nil {
		bindings.Lock()
	}
	// Add the binding to map in memory
	bindings.Add(binding.getKey(), binding)

	// Report metrics
	metrics.Reporter().Report("ConnectionEstablished", 1.0, telemetry.UnitCount)

	return binding
}

type Binding struct {
	session      session.Client
	processor    processor.Processor
	rateLimiter  ratelimiter.RateLimiter
	closedSignal bool
	bindings     *Bindings
}

func (b *Binding) OnBinaryMessage(msg []byte) { b.write(msg) }

func (b *Binding) getKey() string {
	return fmt.Sprintf("%p", b)
}

// Clean up
func (b *Binding) OnClosed(err error) {

	closeChannelSignal := models.MessageInfo{Message: closeMessage, Time: time.Now().UnixNano()}

	// Remove binding from mapping
	if b.bindings != nil {
		b.bindings.Lock()
		b.bindings.Remove(b.getKey())
		b.bindings.Unlock()
	}

	b.closedSignal = true
	b.processor.PutToBuffer(closeChannelSignal)
}
