package test_utils

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/devhub/mdaas-ingest/logger"
	"code.justin.tv/devhub/mdaas-ingest/models"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

func GetTestString(name string, num int) string {
	return fmt.Sprint(name, num)
}

type messageState string

const (
	FirstMessage               messageState = "firstMessage"
	NormalMessage              messageState = "normalMessage"
	UpdatedMessage             messageState = "updatedMessage"
	FirstKeyEqualLastFullState messageState = "firstKeyEqualLastFullState"
	UserTokenFirstMessage      messageState = "userTokenFirstMessage"
)

func GetCloneToBroadcasterMapping(messageState messageState) [][]models.BroadcasterKinesisInfo {
	initialInfo := make([][]models.BroadcasterKinesisInfo, 2)
	initialInfo[0] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2)},
	}
	initialInfo[1] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2)},
	}

	normalInfo := make([][]models.BroadcasterKinesisInfo, 2)
	normalInfo[0] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 1), LastFullState: GetTestString(TestLastFullState, 1)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 2), LastFullState: GetTestString(TestLastFullState, 2)},
	}
	normalInfo[1] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 3), LastFullState: GetTestString(TestLastFullState, 3)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 4), LastFullState: GetTestString(TestLastFullState, 4)},
	}

	updated := make([][]models.BroadcasterKinesisInfo, 2)
	updated[0] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 1), LastFullState: GetTestString(TestLastFullState, 5)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 2), LastFullState: GetTestString(TestLastFullState, 6)},
	}
	updated[1] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 3), LastFullState: GetTestString(TestLastFullState, 7)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 4), LastFullState: GetTestString(TestLastFullState, 8)},
	}

	firstKeyEqualLastFullState := make([][]models.BroadcasterKinesisInfo, 2)
	firstKeyEqualLastFullState[0] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 1), LastFullState: GetTestString(TestFirstKeyFrame, 1)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 2), LastFullState: GetTestString(TestFirstKeyFrame, 2)},
	}
	firstKeyEqualLastFullState[1] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: GetTestString(TestBroadcasterID, 1), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 3), LastFullState: GetTestString(TestFirstKeyFrame, 3)},
		{BroadcasterID: GetTestString(TestBroadcasterID, 2), FirstKeyFrame: GetTestString(TestFirstKeyFrame, 4), LastFullState: GetTestString(TestFirstKeyFrame, 4)},
	}

	userTokenFirstMessageInfo := make([][]models.BroadcasterKinesisInfo, 2)
	userTokenFirstMessageInfo[0] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: TestBroadcasterID},
	}
	userTokenFirstMessageInfo[1] = []models.BroadcasterKinesisInfo{
		{BroadcasterID: TestBroadcasterID},
	}

	switch messageState {
	case FirstMessage:
		return initialInfo
	case NormalMessage:
		return normalInfo
	case UpdatedMessage:
		return updated
	case FirstKeyEqualLastFullState:
		return firstKeyEqualLastFullState
	case UserTokenFirstMessage:
		return userTokenFirstMessageInfo
	default:
		return make([][]models.BroadcasterKinesisInfo, 1)
	}
}

func GetEventFromIngest(ifDelta bool) models.Event {
	data := []interface{}{
		[]string{"skill", "q"},
		[]string{"champion"},
	}

	fullEvent := models.Event{
		Full: &models.FullEvent{
			GameID:   GetTestString(TestGameID, 1),
			Data:     data,
			ClientID: GetTestString(TestClientID, 1),
			Env:      "dev",
		},
	}

	if !ifDelta {
		return fullEvent
	} else {
		return models.Event{
			Delta: &models.DeltaEvent{
				GameID: GetTestString(TestGameID, 1),
				Data:   data,
			},
		}
	}
}

func GetEvent(ifFirst bool, ifDelta bool, broadcasterID string, firstKeyframe string, seqStart string) models.Event {
	data := []interface{}{
		[]string{"skill", "q"},
		[]string{"champion"},
	}

	fullEvent := models.Event{
		Full: &models.FullEvent{
			BroadcasterID: broadcasterID,
			GameID:        GetTestString(TestGameID, 1),
			Data:          data,
			ClientID:      GetTestString(TestClientID, 1),
			Env:           "dev",
			FirstKeyFrame: "",
		},
	}

	if ifFirst {
		return fullEvent
	} else if !ifDelta {
		fullEvent.Full.FirstKeyFrame = firstKeyframe
		return fullEvent
	} else {
		return models.Event{
			Delta: &models.DeltaEvent{
				BroadcasterID: broadcasterID,
				GameID:        GetTestString(TestGameID, 1),
				Data:          data,
				SeqStart:      seqStart,
				FirstKeyFrame: firstKeyframe,
			},
		}
	}
}

func GetKinesisInputs(ifFirst bool, ifDelta bool, clone int, infos [][]models.BroadcasterKinesisInfo) []*kinesis.PutRecordsRequestEntry {
	broadcasters := infos[clone-1]
	expectedEvents := make([]models.Event, len(broadcasters))
	for i, broadcaster := range broadcasters {
		expectedEvents[i] = GetEvent(ifFirst, ifDelta, broadcaster.BroadcasterID, broadcaster.FirstKeyFrame, broadcaster.LastFullState)
	}

	var records []*kinesis.PutRecordsRequestEntry
	for _, e := range expectedEvents {
		jsonBlob, _ := json.Marshal(e)
		record := &kinesis.PutRecordsRequestEntry{
			Data:         jsonBlob,
			PartitionKey: aws.String(e.PartitionKey()),
		}
		records = append(records, record)
	}
	return records
}

// GenerateDebugLogJSON is for generating debug log JSON
func GenerateDebugLogJSON(clientID string, gameID string, broadcasterIDs []string, rawMessage string) (string, error) {
	debugLog := logger.NewDebugLog(clientID, gameID, broadcasterIDs, rawMessage)
	return debugLog.FormatMessage()
}

// SetDebugFlagOnConnectPackMsg is to add debug field into connect pack json message
func SetDebugFlagOnConnectPackMsg(message models.MessageInfo, enableDebugLog bool) (*models.MessageInfo, error) {
	connectMsg := &models.ConnectMsg{}
	err := json.Unmarshal([]byte(message.Message), connectMsg)
	if err != nil {
		return nil, err
	}
	connectMsg.Body.Debug = enableDebugLog
	jsonBody, err := json.Marshal(connectMsg)
	if err != nil {
		return nil, err
	}
	messageInfo := GenerateMessageInfoFromRawMessage(string(jsonBody), message.Time)
	return &messageInfo, nil
}

// MaskTokenInMessageInfo is for masking token in connection raw messaga and output a JSON string and bool to specify
// if the transformation is successful
func MaskTokenInMessageInfo(messageInfo models.MessageInfo) (*string, error) {
	connectMsg := &models.ConnectMsg{}
	err := json.Unmarshal([]byte(messageInfo.Message), connectMsg)
	if err != nil {
		return nil, err
	}
	connectMsg.Body.Token = "***"
	jsonInBytes, err := json.Marshal(connectMsg)
	if err != nil {
		return nil, err
	}
	return aws.String(string(jsonInBytes)), nil
}
