package lifecycle

import (
	"io"
	"os"
	"os/signal"
	"runtime/debug"
	"sort"
	"sync"
	"syscall"
	"time"
)

// Manager allows registration of shutdown hooks that must be completed before
// a program is allowed to exit.
type Manager interface {
	SetOnError(Reporter)
	SetOnPanic(Reporter)

	RegisterHook(interface{}, Hook)                  // inserts hook to be executed (stack ordering)
	ExecuteHook(interface{}) error                   // runs previously registered hook
	RunUntilComplete(func())                         // allows launch of goroutines that must complete before shutdown
	TickUntilClosed(func(), time.Duration) io.Closer // ticks the given function until Close() has been called
	ListenForInterrupt() os.Signal                   // block for an OS signal that says we need to stop execution
	WaitForCompletion(until time.Time)               // wait until all goroutines are complete, the given time elapses
	ExecuteAll() error                               // runs all registered hooks in FILO order
}

type manager struct {
	sigs    chan os.Signal
	onError Reporter
	onPanic Reporter
	hooks   map[interface{}]*entry
	added   int
	mutex   sync.Mutex
	wg      sync.WaitGroup
}

// NewManager creates a new manager
func NewManager() Manager {
	return &manager{
		onError: proxiedDefaultErrorReporter,
		onPanic: proxiedDefaultPanicReporter,
		hooks:   make(map[interface{}]*entry),
	}
}

func (m *manager) SetOnError(r Reporter) {
	m.mutex.Lock()
	m.onError = r
	m.mutex.Unlock()
}

func (m *manager) SetOnPanic(r Reporter) {
	m.mutex.Lock()
	m.onPanic = r
	m.mutex.Unlock()
}

func (m *manager) RegisterHook(key interface{}, h Hook) {
	m.mutex.Lock()
	m.hooks[key] = &entry{key, h, m.added}
	m.added += 1
	m.mutex.Unlock()
}

func (m *manager) ExecuteHook(key interface{}) error {
	m.mutex.Lock()
	onError := m.onError
	entry, ok := m.hooks[key]
	delete(m.hooks, key)
	m.mutex.Unlock()
	if ok {
		return execute(key, entry.hook, onError)
	}
	return nil
}

func (m *manager) ListenForInterrupt() os.Signal {
	sigChan := make(chan os.Signal)
	signal.Notify(sigChan, os.Interrupt, syscall.SIGTERM, syscall.SIGUSR2)
	sig := <-sigChan
	signal.Stop(sigChan)
	return sig
}

func (m *manager) WaitForCompletion(until time.Time) {
	ticker := time.NewTicker(until.Sub(time.Now()))
	waiter := make(chan struct{})
	go func() {
		m.wg.Wait()
		close(waiter)
	}()

	select {
	case _, _ = <-waiter:
	case _, _ = <-ticker.C:
	}

	ticker.Stop()
}

func (m *manager) ExecuteAll() error {
	errors := m.executeAll()
	m.wg.Wait()
	return errors
}

func (m *manager) TickUntilClosed(async func(), period time.Duration) io.Closer {
	m.wg.Add(1)
	closer := newTickCloser(m)
	go func() {
		defer m.onComplete(async)
		defer m.wg.Done()
		ticker := time.NewTicker(period)
		defer ticker.Stop()
		for {
			select {
			case _, _ = <-closer.stop:
				return
			case _, _ = <-ticker.C:
				async()
			}
		}
	}()
	return closer
}

func (m *manager) RunUntilComplete(async func()) {
	m.wg.Add(1)
	go func() {
		defer m.onComplete(async)
		defer m.wg.Done()
		async()
	}()
}

func (m *manager) executeAll() error {
	errs := ErrorList(nil)
	m.mutex.Lock()
	listing := make(entries, len(m.hooks))
	i := 0
	for key := range m.hooks {
		listing[i] = m.hooks[key]
		i = i + 1
	}
	m.hooks = map[interface{}]*entry{}
	m.mutex.Unlock()
	sort.Sort(listing)
	for _, entry := range listing {
		if err := execute(entry.key, entry.hook, m.onError); err != nil {
			if errs == nil {
				errs = NewErrorList(err)
			} else {
				errs.Append(err)
			}
		}
	}
	return errs
}

func (m *manager) onComplete(src interface{}) {
	if r := recover(); r != nil {
		m.mutex.Lock()
		onPanic := m.onPanic
		m.mutex.Unlock()
		onPanic(src, PanicToError(r, debug.Stack()))
	}
}

func execute(key interface{}, hook Hook, reporter Reporter) error {
	if err := hook(); err != nil {
		reporter(key, err)
		return err
	}
	return nil
}
