package dynamo

import (
	"context"
	"errors"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"

	settings "code.justin.tv/devhub/twitch-e2-ingest/http-settings"
	"code.justin.tv/devhub/twitch-e2-ingest/sns"
	"code.justin.tv/video/metrics-middleware/v2/awsmetric"
	"code.justin.tv/video/metrics-middleware/v2/operation"
)

// Allowlist represents the games and clients that are allowed to publish data to E2 Ingest.
//go:generate go run github.com/vektra/mockery/cmd/mockery -name Allowlist
type Allowlist interface {
	ListClients(ctx context.Context, request *ListClientsRequest) ([]*Client, error)
	ListEntriesByOrg(ctx context.Context, request *ListEntriesByOrgRequest) ([]*AllowlistEntry, error)
	ListEntries(ctx context.Context, request *ListEntriesRequest) (*ListEntriesResponse, error)
	ListPages(ctx context.Context, request *ListPagesRequest) (*ListPagesResponse, error)
	IsAllowlisted(ctx context.Context, gameID, clientID string) (bool, error)
	CreateAllowlistEntry(ctx context.Context, request *CreateAllowlistEntryRequest) (*AllowlistEntry, error)
	UpdateAllowlistEntry(ctx context.Context, request *UpdateAllowlistEntryRequest) (*AllowlistEntry, error)
	DeleteAllowlistEntry(ctx context.Context, gameID, clientID string) error
}

type Config struct {
	AWSRegion        string
	RoleARN          string
	TableName        string
	OperationStarter *operation.Starter
}

type allowlist struct {
	dynamo    dynamodbiface.DynamoDBAPI
	tableName string
	sns       sns.Publisher
}

// NewAllowlist instantiates a new Allowlist, backed by DynamoDB.
func NewAllowlist(c *Config, p sns.Publisher) (Allowlist, error) {
	if c.TableName == "" {
		return nil, errors.New("missing table name for Allowlist")
	}

	httpClient, err := settings.NewHTTPClientWithSettings(settings.ClientSettings{
		ConnectTimeout:        1 * time.Second,
		IdleConnTimeout:       90 * time.Second,
		KeepAlive:             30 * time.Second,
		MaxIdleConns:          0,
		MaxIdleConnsPerHost:   100,
		ResponseHeaderTimeout: 1 * time.Second,
		TLSHandshakeTimeout:   1 * time.Second,
	})
	if err != nil {
		return nil, err
	}

	awsConfig := &aws.Config{
		HTTPClient:          httpClient,
		Region:              aws.String(c.AWSRegion),
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	}
	sess, err := session.NewSession(awsConfig)
	if err != nil {
		return nil, err
	}

	if c.OperationStarter != nil {
		metricClient := awsmetric.Client{Starter: c.OperationStarter}
		sess = metricClient.AddToSession(sess)
	}

	var creds *credentials.Credentials
	if c.RoleARN != "" {
		creds = stscreds.NewCredentials(sess, c.RoleARN)
	}

	return &allowlist{
		dynamo:    dynamodb.New(sess, awsConfig.WithCredentials(creds)),
		tableName: c.TableName,
		sns:       p,
	}, nil
}
