package dynamo

import (
	"context"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"

	"code.justin.tv/devhub/twitch-e2-ingest/sns"
)

type CreateAllowlistEntryRequest struct {
	GameID         string
	ClientID       string
	OrganizationID string
	Products       []string
	CreatedBy      string
}

func (a *allowlist) CreateAllowlistEntry(ctx context.Context, r *CreateAllowlistEntryRequest) (*AllowlistEntry, error) {
	topics, err := a.sns.CreateTopics(ctx, r.GameID, r.ClientID)
	if err != nil {
		return nil, err
	}

	const conditionExpression = "attribute_not_exists(GameID) AND attribute_not_exists(ClientID)"
	attributeValues, updateExpression := r.AttributeValuesAndUpdateExpression(time.Now(), topics)
	input := &dynamodb.UpdateItemInput{
		TableName: &a.tableName,
		Key: map[string]*dynamodb.AttributeValue{
			"GameID":   {S: aws.String(r.GameID)},
			"ClientID": {S: aws.String(r.ClientID)},
		},
		ConditionExpression:       aws.String(conditionExpression),
		ReturnValues:              aws.String(dynamodb.ReturnValueAllNew),
		UpdateExpression:          aws.String(updateExpression),
		ExpressionAttributeValues: attributeValues,
	}
	output, err := a.dynamo.UpdateItemWithContext(ctx, input)
	if err != nil {
		var conditionErr *dynamodb.ConditionalCheckFailedException
		if errors.As(err, &conditionErr) {
			return nil, &AlreadyExistsError{
				gameID:   r.GameID,
				clientID: r.ClientID,
			}
		}
		return nil, err
	}

	var entry AllowlistEntry
	if err := dynamodbattribute.UnmarshalMap(output.Attributes, &entry); err != nil {
		return nil, err
	}

	return &entry, nil
}

func (r *CreateAllowlistEntryRequest) AttributeValuesAndUpdateExpression(t time.Time, topics sns.Topics) (map[string]*dynamodb.AttributeValue, string) {
	attributeValues := map[string]*dynamodb.AttributeValue{
		":created_at": {N: aws.String(strconv.FormatInt(t.Unix(), 10))},
		":sns_topics": {SS: aws.StringSlice(topics.ARNs())},
	}
	statements := []string{
		"CreatedAt = :created_at",
		"SNSTopics = :sns_topics",
	}

	if r.OrganizationID != "" {
		attributeValues[":organization_id"] = &dynamodb.AttributeValue{S: aws.String(r.OrganizationID)}
		statements = append(statements, "OrganizationID = :organization_id")
	}
	if len(r.Products) != 0 {
		attributeValues[":products"] = &dynamodb.AttributeValue{SS: aws.StringSlice(r.Products)}
		statements = append(statements, "Products = :products")
	}
	if r.CreatedBy != "" {
		attributeValues[":created_by"] = &dynamodb.AttributeValue{S: aws.String(r.CreatedBy)}
		statements = append(statements, "CreatedBy = :created_by")
	}

	return attributeValues, fmt.Sprintf("SET %s", strings.Join(statements, ", "))
}

type AlreadyExistsError struct {
	gameID   string
	clientID string
}

func (e *AlreadyExistsError) Error() string {
	return fmt.Sprintf("game ID %q and client ID %q already allowlisted", e.gameID, e.clientID)
}
