package dynamo

import (
	"context"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
)

type ListEntriesRequest struct {
	First  int
	After  string
	Filter *ListEntriesFilter
}

type ListEntriesResponse struct {
	Entries   []*AllowlistEntry
	EndCursor string
}

func (a *allowlist) ListEntries(ctx context.Context, r *ListEntriesRequest) (*ListEntriesResponse, error) {
	startKey, err := FromCursor(r.After)
	if err != nil {
		return nil, err
	}

	var limit *int64
	if r.First > 0 {
		limit = aws.Int64(int64(r.First))
	}

	filterExpression, expAttrVal := r.Filter.FilterExprAndExprAttrVals()
	input := &dynamodb.ScanInput{
		ExclusiveStartKey:         startKey,
		Limit:                     limit,
		Select:                    aws.String(dynamodb.SelectAllAttributes),
		TableName:                 aws.String(a.tableName),
		FilterExpression:          filterExpression,
		ExpressionAttributeValues: expAttrVal,
	}
	var items []map[string]*dynamodb.AttributeValue
	err = a.dynamo.ScanPagesWithContext(ctx, input, func(output *dynamodb.ScanOutput, lastPage bool) bool {
		items = append(items, output.Items...)
		if r.First > 0 && len(items) >= r.First {
			return false
		}
		return true
	})
	if err != nil {
		return nil, err
	}

	if r.First > 0 && len(items) > r.First {
		items = items[:r.First]
	}

	var entries []*AllowlistEntry
	if err := dynamodbattribute.UnmarshalListOfMaps(items, &entries); err != nil {
		return nil, err
	}

	cursor := ""
	if len(items) > 0 {
		last := items[len(items)-1]
		lastEvaluatedKey := map[string]*dynamodb.AttributeValue{
			"GameID":   last["GameID"],
			"ClientID": last["ClientID"],
		}
		cursor, err = ToCursor(lastEvaluatedKey)
		if err != nil {
			return nil, err
		}
	}

	return &ListEntriesResponse{
		Entries:   entries,
		EndCursor: cursor,
	}, nil
}
