package interpol

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	settings "code.justin.tv/devhub/twitch-e2-ingest/http-settings"
	"code.justin.tv/devhub/twitch-e2-ingest/models"
	log "github.com/sirupsen/logrus"
)

// InterpolClient provides functions by calling interpol
//go:generate go run github.com/vektra/mockery/cmd/mockery -name InterpolClient
type InterpolClient interface {
	GetClientInfo(token string) (*models.AuthResp, error)
}

// New initializes client
func New(interpolHost string) (InterpolClient, error) {
	httpClient, err := settings.NewHTTPClientWithSettings(settings.ClientSettings{
		ConnectTimeout:        5 * time.Second,
		ExpectContinueTimeout: 1 * time.Second,
		IdleConnTimeout:       90 * time.Second,
		KeepAlive:             30 * time.Second,
		MaxIdleConns:          0,
		MaxIdleConnsPerHost:   100000,
		ResponseHeaderTimeout: 10 * time.Second,
		TLSHandshakeTimeout:   5 * time.Second,
	})

	if err != nil {
		return nil, err
	}

	return &clientImpl{
		interpolHost: interpolHost,
		httpClient:   httpClient,
	}, nil
}

type clientImpl struct {
	interpolHost string
	httpClient   *http.Client
}

func (i *clientImpl) GetClientInfo(token string) (*models.AuthResp, error) {
	var authResult models.AuthResp
	// Currently it is only talking to id.twitch.tv, make it variable if we find needs
	req, err := http.NewRequest("GET", fmt.Sprintf("%s/oauth2/validate", i.interpolHost), nil)
	if err != nil {
		log.Error(err)
		return nil, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", token))

	resp, err := i.sendRequest(req)
	if err != nil {
		return nil, err
	}

	defer resp.Body.Close()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Error(err)
		return nil, err
	}

	err = json.Unmarshal(body, &authResult)
	if err != nil {
		log.Error(err)
		return nil, err
	}

	return &authResult, nil
}

// if needed, expose this method as public
func (i *clientImpl) sendRequest(req *http.Request) (*http.Response, error) {
	resp, err := i.httpClient.Do(req)
	if err != nil {
		log.Error(err)
		return nil, err
	} else if resp == nil {
		return nil, errors.New("failed to talk to identity")
	}

	if resp.StatusCode >= 400 {
		return nil, errors.New("token not associated with client")
	}

	return resp, err
}
