package sns

import (
	"context"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"

	"code.justin.tv/devhub/twitch-e2-ingest/awsmocks"
	settings "code.justin.tv/devhub/twitch-e2-ingest/http-settings"
)

const (
	eventName = "ingest_data"
)

type Message struct {
	Default string `json:"default"`
}

// Publisher publishes an event to an SNS topic, logging any errors that may
// have occurred, and manages SNS topics.
//go:generate go run github.com/vektra/mockery/cmd/mockery -name Publisher
type Publisher interface {
	Publish(interface{}) error
	PublishByARN(data interface{}, arn string) error
	CreateTopic(ctx context.Context, gameID string, clientID string, env TopicEnv) (*Topic, error)
	CreateTopics(ctx context.Context, gameID, clientID string) (Topics, error)
}

// NewPublisher instantiates a new Publisher
func NewPublisher(c *Config) (Publisher, error) {
	httpClient, err := settings.NewHTTPClientWithSettings(settings.ClientSettings{
		ConnectTimeout:        5 * time.Second,
		ExpectContinueTimeout: 1 * time.Second,
		IdleConnTimeout:       90 * time.Second,
		KeepAlive:             30 * time.Second,
		MaxIdleConns:          0,
		MaxIdleConnsPerHost:   100000,
		ResponseHeaderTimeout: 10 * time.Second,
		TLSHandshakeTimeout:   5 * time.Second,
	})
	if err != nil {
		return nil, err
	}

	awsConfig := &aws.Config{
		HTTPClient:          httpClient,
		Region:              aws.String(c.awsRegion()),
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	}
	sess, err := session.NewSession(awsConfig)
	if err != nil {
		return nil, err
	}

	var creds *credentials.Credentials
	if c.RoleARN != "" {
		creds = stscreds.NewCredentials(sess, c.RoleARN)
	}

	return &publisherImpl{
		snsClient: sns.New(sess, awsConfig.WithCredentials(creds)),
		config:    c,
	}, nil
}

type publisherImpl struct {
	snsClient snsiface.SNSAPI
	config    *Config
}

func NewTestPublisher(snsmock *awsmocks.SNSAPI, c *Config) (Publisher, error) {
	return &publisherImpl{snsClient: snsmock, config: c}, nil
}
