package websocket

import (
	"context"
	"net/url"
	"sync/atomic"
	"time"
	"unsafe"

	"code.justin.tv/devhub/lib-lifecycle/src/lifecycle"
	"code.justin.tv/karlpatr/message-prototype/libs/session"
	"github.com/gobwas/ws"
)

func NewClientFactory(url *url.URL, settings *Settings) session.ClientFactory {
	internal := fixup(settings)
	internal.Lifecycle = lifecycle.NewManager()
	if settings.Lifecycle != nil {
		settings.Lifecycle.RegisterHook(settings, func() error { internal.Lifecycle.ExecuteAll(); return nil })
	}
	return func(bind session.Binding) (session.Client, error) {
		return NewClient(url, bind, internal)
	}
}

func NewClientResolver(settings *Settings) session.ClientResolver {
	internal := fixup(settings)
	internal.Lifecycle = lifecycle.NewManager()
	if settings.Lifecycle != nil {
		settings.Lifecycle.RegisterHook(settings, func() error { internal.Lifecycle.ExecuteAll(); return nil })
	}
	return func(url *url.URL, bind session.Binding) (session.Client, error) {
		return NewClient(url, bind, internal)
	}
}

func NewClient(url *url.URL, bind session.Binding, settings *Settings) (session.Client, error) {
	conn, _, _, err := ws.Dialer{TLSConfig: settings.Certs}.Dial(context.Background(), url.String())
	if err != nil {
		return nil, err
	}

	internal := fixup(settings)
	internal.Lifecycle = lifecycle.NewManager()
	linkSettings := newLinkSettings(ws.MaskFrame, time.Second, settings.Logger)

	// shut down the client lifecycle to close; in practice this makes the client
	// sends an appropriate close message to the service which causes graceful
	// termination.
	link := &link{linkSettings, func(error) { go internal.Lifecycle.ExecuteAll() }, conn}

	if settings.Lifecycle != nil {
		// if we're in a managed lifecycle, then remove ourselves as a managed resource on close
		// (this is still a call to shutdown internal under the hood)
		settings.Lifecycle.RegisterHook(internal, func() error {
			return internal.Lifecycle.ExecuteAll()
		})

		// if we're in a managed lifecycle, then remove ourselves as a managed resource on close	 		link.onClose = func(error) {
		// (this is still a call to shutdown safeSettings under the hood)
		link.onClose = func(error) {
			go settings.Lifecycle.ExecuteHook(internal)
		}
	}

	// request client connection close on client shutdown request
	closed := int32(0)
	result := unsafe.Pointer(nil)
	internal.Lifecycle.RegisterHook(bind, func() error {
		var out error
		if pErr := (*error)(atomic.LoadPointer(&result)); pErr != nil {
			out = *pErr
		}
		bind.OnClosed(out)
		return nil
	})

	internal.Lifecycle.RegisterHook(link, func() error {
		if atomic.LoadInt32(&closed) == 0 {
			link.writeClose(ws.StatusNormalClosure, "")
		}
		return nil
	})

	internal.Lifecycle.RunUntilComplete(func() {
		defer func() {
			atomic.StoreInt32(&closed, 1)
			link.Close(err)
		}()
		defer settings.gracefulClose(link.conn)
		value := execute(bind, link, ws.StateClientSide, internal, func() bool { return false })
		atomic.StorePointer(&result, unsafe.Pointer(&value))
	})

	return link, nil
}
