package websocket

import (
	"net"
	"time"

	"code.justin.tv/karlpatr/message-prototype/libs/logging"
	"code.justin.tv/karlpatr/message-prototype/libs/session"
	"github.com/gobwas/ws"
)

type maskFunc func(ws.Frame) ws.Frame
type closeFunc func(error)

type linkSettings struct {
	mask    maskFunc
	timeout time.Duration
	logger  logging.Function
}

type link struct {
	*linkSettings
	onClose closeFunc
	conn    net.Conn
}

var _ session.Client = (*link)(nil)

func newLinkSettings(mask maskFunc, timeout time.Duration, logger logging.Function) *linkSettings {
	return &linkSettings{mask, timeout, logger}
}

func (w *link) Address() net.Addr { return w.conn.RemoteAddr() }

func (w *link) WriteText(content string) error {
	return w.writeFrame(ws.NewTextFrame([]byte(content)))
}

func (w *link) WriteBinaryAsText(content []byte) error {
	return w.writeFrame(ws.NewTextFrame(content))
}

func (w *link) WriteBinary(content []byte) error {
	return w.writeFrame(ws.NewBinaryFrame(content))
}

func (w *link) Close(err error) { w.onClose(err) }

func (w *link) writeClose(code ws.StatusCode, reason string) {
	w.writeFrame(ws.NewCloseFrame(ws.NewCloseFrameBody(code, reason)))
}

func (w *link) writeFrame(frame ws.Frame) error {
	body, err := ws.CompileFrame(w.mask(frame))
	if err != nil {
		return err
	}
	if err = w.conn.SetWriteDeadline(time.Now().Add(w.timeout)); err == nil {
		_, err = w.conn.Write(body) // is thread safe (https://golang.org/pkg/net/#Conn)
	}
	if err != nil {
		w.logger(logging.Debug, "Write error", w.Address(), frame.Header.OpCode, err)
	}
	return err
}
