//Package naiive implements a simple regex based static analysis tool for Go
package naiive // import "code.justin.tv/tshadwell/nice/naiive"

import (
	"bytes"
	"fmt"
	"regexp"
	"text/template"
)

type Confidence uint

const (
	ZeroConfidence Confidence = iota
	LowConfidence
	MediumConfidence
	HighConfidence
)

type Severity uint

const (
	ZeroSeverity Severity = iota
	LowSeverity
	MediumSeverity
	HighSeverity
)

// Test represents a single naiive analysis
type Test struct {
	*regexp.Regexp

	Name       string
	Confidence Confidence
	Severity   Severity

	Desc        string
	Examples    []string
	NonExamples []string
	ShortName   string

	// a format for constructing
	// longer examples for each of the examples
	ExampleFormat string
	Category      string
}

type ExampleInfo struct {
	// "main"
	PackageName string

	// one of the examples from the 'Examples' array
	Example string

	// an end of line comment
	FlaggedLine string
}

func (t Test) example(inf ExampleInfo) (example string, err error) {
	var b bytes.Buffer
	var tmpl *template.Template
	if tmpl, err = template.New(fmt.Sprintf("test-%s", t.Name)).Parse(
		t.ExampleFormat); err != nil {
		return
	}

	if err = tmpl.Execute(&b, inf); err != nil {
		return
	}

	return b.String(), nil

}

func (t Test) Example() (example string, err error) {
	return t.example(ExampleInfo{
		PackageName: "example",
		Example:     t.Examples[0],
		FlaggedLine: " // error on this line!",
	})
}

// TODO: replace with actual test name
func (t Test) flag() string { return "//want `" + t.Name + "`" }

func (t Test) AnalysisTests(packageName string) (tests []string, err error) {
	tests = make([]string, len(t.Examples)+len(t.NonExamples))
	for i, example := range t.Examples {
		if tests[i], err = t.example(ExampleInfo{
			PackageName: packageName,
			Example:     example,
			FlaggedLine: t.flag(),
		}); err != nil {
			return
		}
	}

	for i, nonExample := range t.NonExamples {
		if tests[len(t.Examples)+i], err = t.example(ExampleInfo{
			PackageName: "main",
			Example:     nonExample,
			FlaggedLine: "",
		}); err != nil {
			return
		}
	}

	return
}

type TestCaseInput struct {
	Example string
	TestCaseInfo
}

type TestCaseInfo interface {
	PackageName() string
	FlaggedLine() string
}

func (t Test) TestCases(testInfo TestCaseInfo) (code []string, err error) {
	var b bytes.Buffer
	var tmpl *template.Template
	if tmpl, err = template.New(fmt.Sprintf("test-%s", t.Name)).Parse(t.ExampleFormat); err != nil {
		return
	}

	for _, example := range t.Examples {
		if err = tmpl.Execute(&b, TestCaseInput{
			Example:      example,
			TestCaseInfo: testInfo,
		}); err != nil {
			return
		}
		code = append(code, b.String())
		b.Reset()
	}

	return
}
