// import { Configuration, Data, IDataSource, TokenExpiredFn, TokenRefreshFn } from 'enhanced-experiences-sdk';
import * as mdaas from 'enhanced-experiences-sdk';
import { EventEmitter } from 'events';
import { Agent } from 'https';
import LCUConnector from 'lcu-connector';
import fetch from 'node-fetch';
import { SidecarClient } from '../base';
import { LCUMetadata } from '../models/lcu-metadata';
import { AllGameData } from '../models/league-api/all-game-data';
import { LeagueOfLegendsDataCache } from '../models/league-data-cache';
import { MDaaSPayload } from './../models/mdaas-payload';
import {
  ACTIVEPLAYER_URL,
  ACTIVEPLAYERABILITIES_URL,
  ACTIVEPLAYERNAME_URL,
  ACTIVEPLAYERRUNES_URL,
  ALLGAMEDATA_URL,
  EVENTDATA_URL,
  GAMESTATS_URL,
  HOSTNAME,
  LCU_CLIENT_INFO,
  LCU_READY_URL,
  LOLGAMEID,
  PLAYERITEMS_URL,
  PLAYERLIST_URL,
  PLAYERMAINRUNES_URL,
  PLAYERSCORES_URL,
  PLAYERSUMMONERSPELLS_URL,
  PORT
} from './constants';
import { createAbilityHistory } from './transforms/ability-history';
import { createItemHistory } from './transforms/item-history';
import { createMDaaSPayload } from './transforms/payload';
import { createPlayerMap } from './transforms/player-map';

const tokenExpiredStub: mdaas.TokenExpiredFn = (_fn: mdaas.TokenRefreshFn) => {
  console.error('unexpected token refresh call-back');
  return false;
};

export interface LeagueClientOptions {
  pollingInterval: number;
  broadcasterID: string;
  appTokenID: string;
}

const DEFAULT_OPTIONS: LeagueClientOptions = {
  pollingInterval: 1000,
  broadcasterID: '',
  appTokenID: ''
};

export class LeagueOfLegendsSidecarClient extends SidecarClient {
  // public ActivePlayerName: string | undefined;
  public MDaaSConnection = mdaas.default();
  public dataFlowRunning: boolean = false;

  public lcuConnector: LCUConnector;
  public lcuRunning: boolean = false;
  public lcuMetadata?: LCUMetadata;
  public lcuReady: boolean = false;
  public config!: mdaas.Configuration;
  public configOptions: LeagueClientOptions;
  public activeSessionId: string = '';

  private lcuEventEmitter: EventEmitter;
  private isDataCacheValid: boolean = false;
  private dataCache: LeagueOfLegendsDataCache;

  private connectionLock = false;

  private leagueApiPollTimer?: NodeJS.Timer;

  constructor(options: LeagueClientOptions = DEFAULT_OPTIONS) {
    super(PORT, HOSTNAME);
    this.configOptions = Object.assign({}, DEFAULT_OPTIONS, options);
    this.lcuConnector = new LCUConnector('');
    this.lcuEventEmitter = new EventEmitter();
    this.dataCache = this.buildCacheObject();

    this.initializeLCUConnector();
    this.startLCUConnector();
    this.leagueApiPollTimer = undefined;
  }

  public initializeLCUConnector() {
    this.lcuConnector.on('connect', (data: LCUMetadata) => {
      this.lcuMetadata = data;
      this.lcuRunning = true;
      const lcuReadyCheckInterval = setInterval(async () => {
        try {
          const isReady = await this.lcuReadyCheck();
          if (isReady) {
            this.lcuEventEmitter.emit('ready');
            clearInterval(lcuReadyCheckInterval);
          }
        } catch (e) {
          console.log(e);
        }
      }, 200);
      this.lcuReadyCheck();
    });

    this.lcuConnector.on('disconnect', () => {
      this.lcuRunning = false;
      this.lcuMetadata = undefined;
      this.dataCache.region = undefined;
    });

    this.lcuEventEmitter.on('ready', () => {
      this.lcuGetData();
    });
  }

  public startLCUConnector() {
    this.lcuConnector.start();
  }

  public stopLCUConnector() {
    this.lcuConnector.stop();
  }

  public async lcuReadyCheck() {
    const agent = this.setupAgent();
    if (this.lcuRunning && this.lcuMetadata) {
      const lcuBaseUrl = `${this.lcuMetadata.protocol}://${this.lcuMetadata.address}:${this.lcuMetadata.port}`;

      const isReadyResponse = await fetch(LCU_READY_URL(lcuBaseUrl), {
        agent,
        headers: {
          Authorization: `Basic ${Buffer.from(`${this.lcuMetadata.username}:${this.lcuMetadata.password}`).toString(
            'base64'
          )}`
        }
      });

      if (!isReadyResponse.ok) {
        throw new Error('Ready call failed');
      }

      const readyResponseText = await isReadyResponse.text();
      return readyResponseText === 'true';
    }
    return false;
  }

  public async lcuGetData() {
    const agent = this.setupAgent();
    if (this.lcuReadyCheck && this.lcuMetadata) {
      const lcuBaseUrl = `${this.lcuMetadata.protocol}://${this.lcuMetadata.address}:${this.lcuMetadata.port}`;
      const platformResponse = await fetch(LCU_CLIENT_INFO(lcuBaseUrl), {
        agent,
        headers: {
          Authorization: `Basic ${Buffer.from(`${this.lcuMetadata.username}:${this.lcuMetadata.password}`).toString(
            'base64'
          )}`
        }
      });

      if (platformResponse.ok) {
        const body = await platformResponse.json();
        this.dataCache.region = body.LoginDataPacket.platformId;
      }
    }
  }

  public async isResponsive(): Promise<boolean> {
    const agent = this.setupAgent();
    const response = await fetch(ACTIVEPLAYERNAME_URL, { agent });
    return response.ok;
  }

  public async getDataFromLoLClient(endPointURL: string, summonerName: string = '') {
    const agent = this.setupAgent();
    const response = await fetch(`${endPointURL}${summonerName}`, { agent });
    return await response.json();
  }

  public async getGameInfo() {
    const gameData: AllGameData = await this.getAllGameData();

    if (this.dataCache.region === undefined) {
      await this.lcuReadyCheck();
      await this.lcuGetData();
    }
    this.isDataCacheValid = this.isPayloadGood(gameData);

    if (this.isDataCacheValid) {
      this.dataCache.gameData = gameData;
      this.appendItemHistory(gameData);
      this.appendAbilityHistory(gameData);
      if (gameData.allPlayers) {
        this.dataCache.allPlayers = createPlayerMap(gameData.allPlayers, this.dataCache.allPlayers);
      }
    }
  }

  // arguementless calls
  public async getActivePlayer(): Promise<string> {
    return this.getDataFromLoLClient(ACTIVEPLAYER_URL);
  }
  public async getActivePlayerAbilities(): Promise<string> {
    return this.getDataFromLoLClient(ACTIVEPLAYERABILITIES_URL);
  }
  public async getActivePlayerName(): Promise<string> {
    return this.getDataFromLoLClient(ACTIVEPLAYERNAME_URL);
  }
  public async getActivePlayerRunes(): Promise<string> {
    return this.getDataFromLoLClient(ACTIVEPLAYERRUNES_URL);
  }
  public async getAllGameData(): Promise<AllGameData> {
    return this.getDataFromLoLClient(ALLGAMEDATA_URL);
  }
  public async getEvents(): Promise<string> {
    return this.getDataFromLoLClient(EVENTDATA_URL);
  }
  public async getGameStats(): Promise<string> {
    return this.getDataFromLoLClient(GAMESTATS_URL);
  }
  public async getPlayerList(): Promise<string> {
    return this.getDataFromLoLClient(PLAYERLIST_URL);
  }

  // calls that take a single arguement of summonerName
  public async getPlayerMainRunes(summonerName: string): Promise<string> {
    return this.getDataFromLoLClient(PLAYERMAINRUNES_URL, summonerName);
  }
  public async getPlayerSummonerSpells(summonerName: string): Promise<string> {
    return this.getDataFromLoLClient(PLAYERSUMMONERSPELLS_URL, summonerName);
  }
  public async getPlayerItems(summonerName: string): Promise<string> {
    return this.getDataFromLoLClient(PLAYERITEMS_URL, summonerName);
  }
  public async getPlayerScores(summonerName: string): Promise<string> {
    return this.getDataFromLoLClient(PLAYERSCORES_URL, summonerName);
  }
  // end api calls

  public async setupMDaaSConnection(initialPayload: mdaas.Data) {
    this.config = {
      token: this.configOptions.appTokenID,
      broadcasterIds: [],
      gameId: LOLGAMEID,
      environment: 'prod',
      onTokenExpired: tokenExpiredStub,
      initialData: initialPayload,
      timeoutMs: 5000,
      debugFn: process.env.TWITCH_LOL_SIDECAR_DEBUG === '1'
    };
  }

  public async disconnect() {
    this.MDaaSConnection.disconnect();
    this.activeSessionId = '';
    this.dataFlowRunning = false;
    this.connectionLock = false;
  }

  public async sendAllGameData(): Promise<void> {
    const payload = createMDaaSPayload(this.dataCache);
    await this.MDaaSConnection.updateData(payload);
  }

  public stop() {
    if (this.leagueApiPollTimer) {
      clearInterval(this.leagueApiPollTimer);
      this.leagueApiPollTimer = undefined;
      this.disconnect();
    }
  }

  public start(updateSessionId?: Function) {
    if (!this.leagueApiPollTimer) {
      this.dataCache = this.buildCacheObject();
      this.startDataFlow(updateSessionId);
    }
  }

  public async startDataFlow(updateSessionId?: Function) {
    this.dataFlowRunning = true;
    if (!this.configOptions.appTokenID) {
      console.error('No token available');
    }
    let connectionRetry = 0;
    this.leagueApiPollTimer = setInterval(async () => {
      try {
        if (this.connectionLock || !this.dataFlowRunning) {
          return;
        }
        await this.getGameInfo();
        if (this.isDataCacheValid) {
          this.connectionLock = true;
          const payload = createMDaaSPayload(this.dataCache);
          if (!this.activeSessionId) {
            await this.setupMDaaSConnection(payload);
            this.activeSessionId = await this.MDaaSConnection.connect(this.config);
            if (this.activeSessionId && updateSessionId !== undefined) {
              // if e2 sdk set an active session id lets mark us as connection established and if there is a relevent function to callback lets do so here.
              updateSessionId();
            } else {
              connectionRetry++;
              if (connectionRetry > 10) {
                this.attemptClearConnection();
                connectionRetry = 0;
              }
            }
          } else {
            this.MDaaSConnection.updateData(payload);
          }
          this.connectionLock = false;
        }
      } catch (e) {
        console.error(e);
      }
    }, this.configOptions.pollingInterval);
  }
  private attemptClearConnection() {
    this.dataCache = this.buildCacheObject();
    this.MDaaSConnection.disconnect();
    this.activeSessionId = '';
  }

  private isPayloadGood(payload: mdaas.Data | undefined) {
    return payload && payload.allPlayers && payload.allPlayers.length > 0;
  }

  private appendItemHistory(gameData: AllGameData) {
    if (
      gameData !== undefined &&
      gameData.gameData !== undefined &&
      gameData.activePlayer !== undefined &&
      gameData.activePlayer.summonerName !== undefined &&
      gameData.allPlayers !== undefined
    ) {
      const activePlayerSummonerName = gameData.activePlayer.summonerName;
      const currentTime = gameData.gameData.gameTime;
      const activePlayerIndex: number = gameData.allPlayers.findIndex(p => p.summonerName === activePlayerSummonerName);
      const newItems = gameData.allPlayers[activePlayerIndex].items;
      if (newItems !== undefined && currentTime !== undefined) {
        this.dataCache.itemHistory = createItemHistory(
          newItems,
          this.dataCache.currentItems,
          this.dataCache.itemHistory,
          currentTime
        );
        this.dataCache.currentItems = newItems;
      } else {
        console.log(`Issue starting item history creation. newItems: ${newItems} || currentTime: ${currentTime}`);
      }
    } else {
      console.log(`Issue finding data to create item history.`);
    }
  }

  private appendAbilityHistory(gameData: AllGameData) {
    if (
      gameData !== undefined &&
      gameData.gameData !== undefined &&
      gameData.activePlayer !== undefined &&
      gameData.activePlayer.abilities
    ) {
      const currentTime = gameData.gameData.gameTime;
      const newAbilities = gameData.activePlayer.abilities;
      if (newAbilities !== undefined && currentTime !== undefined) {
        this.dataCache.abilityHistory = createAbilityHistory(
          newAbilities,
          this.dataCache.currentAbilities,
          this.dataCache.abilityHistory,
          currentTime
        );
        this.dataCache.currentAbilities = newAbilities;
      } else {
        console.log(
          `Issue starting ability/skill history creation. newItems: ${newAbilities} || currentTime: ${currentTime}`
        );
      }
    } else {
      console.log(`Issue finding data to create ability/skill history.`);
    }
  }

  private setupAgent() {
    return new Agent({
      rejectUnauthorized: false
    });
  }

  private buildCacheObject(): LeagueOfLegendsDataCache {
    const cache = {
      abilityHistory: [],
      itemHistory: [],
      currentAbilities: { Q: undefined, W: undefined, E: undefined, Passive: undefined, R: undefined },
      currentItems: [],
      allPlayers: {}
    };

    return cache;
  }
}
