package main

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"testing"

	"code.justin.tv/devhub/mdaas-tags-translator/models"
	"github.com/aws/aws-lambda-go/events"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"github.com/stretchr/testify/assert"

	pmock "code.justin.tv/devhub/mdaas-tags-translator/processor/mocks"
)

func TestHandleRequest(t *testing.T) {
	p := &pmock.Processor{}
	logger, hook := test.NewNullLogger()

	t.Run("Successfully processes messages with no error logs", func(t *testing.T) {

		gameData := models.GameData{
			BroacastersID: []string{"TESTUSER1"},
			Data:          getJSON(),
		}

		snsEvent := events.SNSEvent{
			Records: []events.SNSEventRecord{
				{
					SNS: getMessage(gameData),
				},
				{
					SNS: getMessage(gameData),
				},
			},
		}

		p.On("Process", gameData).Return(nil).Twice()

		handleRequest(context.Background(), logger, p, snsEvent)

		assert.Equal(t, 0, len(hook.Entries))
		hook.Reset()

		p.AssertExpectations(t)
	})

	t.Run("Logs errors if process fails", func(t *testing.T) {
		gameData := models.GameData{
			BroacastersID: []string{"TESTUSER2"},
			Data:          getJSON(),
		}

		snsEvent := events.SNSEvent{
			Records: []events.SNSEventRecord{
				{
					SNS: getMessage(gameData),
				},
				{
					SNS: getMessage(gameData),
				},
			},
		}

		p.On("Process", gameData).Return(errors.New("failed")).Twice()

		handleRequest(context.Background(), logger, p, snsEvent)

		assert.Equal(t, 2, len(hook.Entries))
		assert.Equal(t, logrus.ErrorLevel, hook.LastEntry().Level)
		assert.Equal(t, "failed", hook.LastEntry().Message)
		hook.Reset()

		p.AssertExpectations(t)
	})
}

func getMessage(gameData models.GameData) events.SNSEntity {
	message, _ := json.Marshal(gameData)
	return events.SNSEntity{
		Message: string(message),
	}
}

func getJSON() string {
	jsonFile, err := os.Open("./data/data.json")

	if err != nil {
		fmt.Println(err)
	}
	defer jsonFile.Close()
	byteValue, _ := ioutil.ReadAll(jsonFile)

	return string(byteValue)
}
