package main

import (
	"bytes"
	"code.justin.tv/devhub/mdaas-tags-translator/cmd/tag_to_uuid_generator.go/internal"
	"code.justin.tv/devhub/mdaas-tags-translator/data"
	"encoding/json"
	"go/format"
	"io/ioutil"
	"log"
	"os"
	"strings"
	"text/template"
)

type Champions struct {
	Data map[string]interface{} `json:"data"`
}

type Tag struct {
	Name           string `json:"English Localization"`
	Type           string
	ProdTagUUID    string `json:"Production Tag UUID"`
	StagingTagUUID string `json:"Staging Tag UUID"`
}

type Map struct {
	ID    string `json:"mapId"`
	Name  string `json:"mapName"`
	Notes string `json:"notes"`
}

type TagArgs struct {
	Tags map[string]Tag
}

const (
	championType = "Champion"
	roleType     = "Role"
	mapType      = "Map"
)

func main() {
	// Tags
	tagData, err := data.Asset("data/data/tags.json")
	if err != nil {
		log.Fatal("cant load tag data")
	}

	var tags []Tag
	err = json.Unmarshal(tagData, &tags)
	if err != nil {
		log.Fatal("cant parse tag data")
	}

	// Tag Map is a tag map struct with expected raw string as key
	// And actual prod and staging tag as values (in a struct)
	tagMap := map[string]Tag{}
	for _, tag := range tags {
		val := strings.Split(tag.Name, ": ")
		if len(val) != 2 {
			log.Fatalf("failed to parse bad tags %s", tag.Name)
		}
		parsedTag := Tag{
			Name:           val[1],
			Type:           val[0],
			ProdTagUUID:    tag.ProdTagUUID,
			StagingTagUUID: tag.StagingTagUUID,
		}

		tagMap[val[1]] = parsedTag
	}

	// build champion data
	championData, err := data.Asset("data/data/champion.json")
	if err != nil {
		log.Fatal("cant load champion data")
	}

	// Data from champion.json
	var champions Champions
	err = json.Unmarshal(championData, &champions)
	if err != nil {
		log.Fatal("cant parse champion data")
	}

	// build champion data
	mapData, err := data.Asset("data/data/map.json")
	if err != nil {
		log.Fatal("cant load map data")
	}

	var maps []Map
	err = json.Unmarshal(mapData, &maps)
	if err != nil {

	}

	mapRef := make(map[string]Map)
	for _, m := range maps {
		mapRef[m.Name] = m
	}

	var roles map[string][]string
	roleData, err := data.Asset("data/data/positions.json")
	if err != nil {
		log.Fatal("cant load role data")
	}

	err = json.Unmarshal(roleData, &roles)
	if err != nil {
		log.Fatal("cant parse role data")
	}

	for _, tag := range tagMap {
		// Verify this champion key is in the official riot json
		if tag.Type == championType {
			if _, ok := champions.Data[tag.Name]; !ok {
				log.Printf("champion %s could not be found in official reference\n", tag.Name)
			}
		} else if tag.Type == mapType {
			// Verify this map key in the official riot json
			if _, ok := mapRef[tag.Name]; !ok {
				log.Printf("map %s could not be found in official reference\n", tag.Name)
			}
		} else if tag.Type == roleType {
			// Verify the role key in pre-defined json
			if val, ok := roles[tag.Name]; ok {
				for _, roleName := range val {
					roleTag := Tag{
						Name:           roleName,
						Type:           roleType,
						ProdTagUUID:    tag.ProdTagUUID,
						StagingTagUUID: tag.StagingTagUUID,
					}
					// Add the role variances to tag matches
					tagMap[roleName] = roleTag
				}
			} else {
				log.Printf("role %s could not be found in official reference\n", tag.Name)
			}
		}
	}

	var stagingBuf bytes.Buffer
	internal.Must(stagingTagsTemplate.Execute(&stagingBuf, &TagArgs{tagMap}), "stagingTagsTemplate.Execute")
	body, err := format.Source(stagingBuf.Bytes())
	internal.Must(err, "staging tags format.Source")

	pwd, _ := os.Getwd()
	internal.Must(ioutil.WriteFile(pwd+"/data/staging-tags.gen.go", body, 0644), "write staging-tags.gen.go")

	var prodBuf bytes.Buffer
	internal.Must(prodTagsTemplate.Execute(&prodBuf, &TagArgs{tagMap}), "prodTagsTemplate.Execute")
	body, err = format.Source(prodBuf.Bytes())
	internal.Must(err, "prod tags format.Source")

	internal.Must(ioutil.WriteFile(pwd+"/data/prod-tags.gen.go", body, 0644), "prod-tags.gen.go")
}

var stagingTagsTemplate = template.Must(template.New("staging-tags").Parse(`
package data

var StagingTags = map[string]string{
	{{range .Tags}}
		"{{.Name}}": "{{.StagingTagUUID}}",
	{{end}}
}
`))

var prodTagsTemplate = template.Must(template.New("prod-tags").Parse(`
package data

var ProdTags = map[string]string{
	{{range .Tags}}
		"{{.Name}}": "{{.ProdTagUUID}}",
	{{end}}
}
`))
