package http_settings

import (
	"net"
	"net/http"
	"time"

	"golang.org/x/net/http2"
)

type ClientSettings struct {
	ConnectTimeout        time.Duration
	KeepAlive             time.Duration
	ExpectContinueTimeout time.Duration
	IdleConnTimeout       time.Duration
	MaxIdleConns          int
	MaxIdleConnsPerHost   int
	ResponseHeaderTimeout time.Duration
	TLSHandshakeTimeout   time.Duration
}

func NewHTTPClientWithSettings(httpSettings ClientSettings) (*http.Client, error) {
	tr := &http.Transport{
		ResponseHeaderTimeout: httpSettings.ResponseHeaderTimeout,
		Proxy:                 http.ProxyFromEnvironment,
		DialContext: (&net.Dialer{
			KeepAlive: httpSettings.KeepAlive,
			Timeout:   httpSettings.ConnectTimeout,
		}).DialContext,
		MaxIdleConns:          httpSettings.MaxIdleConns,
		IdleConnTimeout:       httpSettings.IdleConnTimeout,
		TLSHandshakeTimeout:   httpSettings.TLSHandshakeTimeout,
		MaxIdleConnsPerHost:   httpSettings.MaxIdleConnsPerHost,
		ExpectContinueTimeout: httpSettings.ExpectContinueTimeout,
	}

	// So client makes HTTP/2 requests
	err := http2.ConfigureTransport(tr)
	if err != nil {
		return nil, err
	}

	return &http.Client{
		Transport: tr,
	}, nil
}
