package jsonparser

import (
	"encoding/json"
	"errors"
	"strconv"
)

type JsonParser interface {
	GetFromPath(...string) (string, error)
}

type jsonParser struct {
	Data interface{}
}

// NewJsonParser converts json into a struct so it can be parsed.
// It can accept a json array or a json object
func NewJsonParser(jsonString string) (JsonParser, error) {
	var dataMap = map[string]interface{}{}

	err := json.Unmarshal([]byte(jsonString), &dataMap)
	if err == nil {
		return &jsonParser{Data: dataMap}, nil
	}

	if _, ok := err.(*json.UnmarshalTypeError); !ok {
		return nil, err
	}

	var dataArray = []interface{}{}

	err = json.Unmarshal([]byte(jsonString), &dataArray)
	if err == nil {
		return &jsonParser{Data: dataArray}, nil
	}

	return nil, err
}

// GetFromPath gets the value at path of data.
// If the resolved value is undefined, a nil string is returned in its place along with an error.
func (p *jsonParser) GetFromPath(pathNodes ...string) (string, error) {
	return deepGet(p.Data, pathNodes)
}

func deepGet(data interface{}, pathNodes []string) (string, error) {
	if len(pathNodes) == 0 {
		if value, ok := data.(string); ok {
			return value, nil
		}

		return "", errors.New("Not found")
	}

	if m, ok := data.(map[string]interface{}); ok {
		return deepGet(m[pathNodes[0]], pathNodes[1:])
	}

	if m, ok := data.([]interface{}); ok {
		index, err := strconv.Atoi(pathNodes[0])
		if err != nil || index >= len(m) {
			return "", errors.New("Not found")
		}

		return deepGet(m[index], pathNodes[1:])
	}

	return "", errors.New("Not found")
}
