package jsonparser

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

var testMap = map[string]interface{}{
	"a": map[string]interface{}{
		"b": map[string]interface{}{
			"c": "test",
		},
	},
}

var testArray = []interface{}{
	map[string]interface{}{
		"a": map[string]interface{}{
			"b": map[string]interface{}{
				"c": "test",
			},
		},
	},
}

var testMixedMapArray = map[string]interface{}{
	"a": map[string]interface{}{
		"b": []interface{}{
			map[string]interface{}{
				"c": "x",
			},
			map[string]interface{}{
				"d": "y",
			},
			map[string]interface{}{
				"e": "z",
			},
		},
	},
}

func TestDeepGet(t *testing.T) {
	t.Run("should be able to get value of path from map struct without error", func(t *testing.T) {
		value, err := deepGet(testMap, []string{
			"a",
			"b",
			"c",
		})
		assert.NoError(t, err)
		assert.Equal(t, testMap["a"].(map[string]interface{})["b"].(map[string]interface{})["c"],
			value)

		value, err = deepGet(testArray, []string{
			"0",
			"a",
			"b",
			"c",
		})
		assert.NoError(t, err)
		assert.Equal(t, testArray[0].(map[string]interface{})["a"].(map[string]interface{})["b"].(map[string]interface{})["c"],
			value)

		value, err = deepGet(testMixedMapArray, []string{
			"a",
			"b",
			"1",
			"d",
		})
		assert.NoError(t, err)
		assert.Equal(t, testMixedMapArray["a"].(map[string]interface{})["b"].([]interface{})[1].(map[string]interface{})["d"],
			value)
	})
	t.Run("should return error if path does not resolve in string", func(t *testing.T) {
		value, err := deepGet(testMap, []string{
			"a",
			"b",
		})
		assert.Error(t, err)
		assert.Equal(t, "", value)

		value, err = deepGet(testMap, []string{
			"a",
			"b",
			"c",
			"d",
		})
		assert.Error(t, err)
		assert.Equal(t, "", value)

		value, err = deepGet(testArray, []string{
			"0",
			"a",
		})
		assert.Error(t, err)
		assert.Equal(t, "", value)

		value, err = deepGet(testMixedMapArray, []string{
			"a",
			"b",
			"3",
			"d",
		})
		assert.Error(t, err)
		assert.Equal(t, "", value)
	})
}

func TestGetFromPath(t *testing.T) {
	jsonMap := `{
		"a": {
			"b": {
				"c": "test"
			}
		}
	}`
	jsonArray := `[{
		"a": {
			"b": {
				"c": "test"
			}
		}
	}]`
	jsonParserMap, _ := NewJsonParser(jsonMap)
	jsonParserArray, _ := NewJsonParser(jsonArray)
	t.Run("should be able to get value of path from map struct without error", func(t *testing.T) {
		value, err := jsonParserMap.GetFromPath("a", "b", "c")
		assert.NoError(t, err)
		assert.Equal(t, "test", value)
	})

	t.Run("should be able to get value of path from array struct without error", func(t *testing.T) {
		value, err := jsonParserArray.GetFromPath("0", "a", "b", "c")
		assert.NoError(t, err)
		assert.Equal(t, "test", value)
	})
}
