package processor

import (
	"context"
	"encoding/json"
	"fmt"

	"code.justin.tv/devhub/mdaas-tags-translator/internal/clients/redis"
	"code.justin.tv/devhub/mdaas-tags-translator/jsonparser"
	"code.justin.tv/devhub/mdaas-tags-translator/models"
	"github.com/sirupsen/logrus"
)

//go:generate mockery -name Processor
type Processor interface {
	Process(models.GameData) error
}

type ProcessorImpl struct {
	Redis  redis.RedisClient
	Logger *logrus.Logger
}

func NewProcessor(r redis.RedisClient, logger *logrus.Logger) *ProcessorImpl {
	return &ProcessorImpl{
		Logger: logger,
		Redis:  r,
	}
}

func (p ProcessorImpl) Process(gameData models.GameData) error {
	parser, err := jsonparser.NewJsonParser(gameData.Data)
	if err != nil {
		return err
	}

	valuesMap := map[string]string{}

	for _, DynamicFieldValue := range getDynamicFieldValues() {
		stringPath, err := pathToString(DynamicFieldValue.path, nil)
		if err != nil {
			p.Logger.Error(err)
		}

		value, err := parser.GetFromPath(stringPath...)
		if err == nil {
			valuesMap[DynamicFieldValue.valueName] = value
		} else {
			p.Logger.Error(err)
		}
	}

	var tags []string
	for _, field := range getFields() {
		stringPath, err := pathToString(field.path, valuesMap)
		if err != nil {
			p.Logger.Warn(err)
		}

		value, err := parser.GetFromPath(stringPath...)
		if err == nil {
			if tag, ok := field.valueTags[value]; ok {
				tags = append(tags, tag)
			} else {
				p.Logger.Warn(fmt.Errorf(`Missing tag for "%s"`, value))
			}
		} else {
			p.Logger.Error(err)
		}
	}

	tagsBytes, err := json.Marshal(tags)
	if err != nil {
		return err
	}

	tagsString := string(tagsBytes)

	for _, broadcasterID := range gameData.BroacastersID {
		savedTagString, found := p.Redis.Get(context.Background(), broadcasterID)
		if found && savedTagString == tagsString {
			continue
		}

		err = p.Redis.Set(context.Background(), broadcasterID, tagsString)
		if err != nil {
			p.Logger.Error(err)
		}
		p.Logger.Info(fmt.Sprintf("Updated channel %s", broadcasterID))
	}
	return nil

}

func pathToString(pathNodes []PathNode, values map[string]string) ([]string, error) {
	var strings []string
	for _, pathNode := range pathNodes {

		value := pathNode.key

		if pathNode.isRef {
			if substituteValue, ok := values[pathNode.key]; ok {
				value = substituteValue
			} else {
				return []string{}, fmt.Errorf("path can not be resolved. Pathnode reference key `%s` is not referencing an existing value", pathNode.key)
			}
		}

		strings = append(strings, value)
	}
	return strings, nil
}

func getFields() []FieldPath {
	return []FieldPath{
		{
			path: []PathNode{
				{key: "allPlayers"},
				{key: "playerName", isRef: true},
				{key: "rawChampionName"},
			},
			valueTags: map[string]string{
				"game_character_displayname_Classic":      "c122215b-a029-4a32-b812-c263c0268310",
				"game_character_displayname_Aatrox":       "",
				"game_character_displayname_Ahri":         "",
				"game_character_displayname_Akali":        "",
				"game_character_displayname_Alistar":      "",
				"game_character_displayname_Amumu":        "",
				"game_character_displayname_Anivia":       "",
				"game_character_displayname_Annie":        "",
				"game_character_displayname_Ashe":         "",
				"game_character_displayname_AurelionSol":  "",
				"game_character_displayname_Azir":         "",
				"game_character_displayname_Bard":         "",
				"game_character_displayname_Blitzcrank":   "",
				"game_character_displayname_Brand":        "df91ccfb-2e7c-4b41-a747-14579bcb8c7d",
				"game_character_displayname_Braum":        "",
				"game_character_displayname_Caitlyn":      "",
				"game_character_displayname_Camille":      "",
				"game_character_displayname_Cassiopeia":   "",
				"game_character_displayname_Chogath":      "",
				"game_character_displayname_Corki":        "",
				"game_character_displayname_Darius":       "",
				"game_character_displayname_Diana":        "",
				"game_character_displayname_Draven":       "",
				"game_character_displayname_DrMundo":      "",
				"game_character_displayname_Ekko":         "",
				"game_character_displayname_Elise":        "",
				"game_character_displayname_Evelynn":      "",
				"game_character_displayname_Ezreal":       "",
				"game_character_displayname_Fiddlesticks": "",
				"game_character_displayname_Fiora":        "",
				"game_character_displayname_Fizz":         "",
				"game_character_displayname_Galio":        "",
				"game_character_displayname_Gangplank":    "",
				"game_character_displayname_Garen":        "",
				"game_character_displayname_Gnar":         "",
				"game_character_displayname_Gragas":       "",
				"game_character_displayname_Graves":       "",
				"game_character_displayname_Hecarim":      "",
				"game_character_displayname_Heimerdinger": "",
				"game_character_displayname_Illaoi":       "",
				"game_character_displayname_Irelia":       "",
				"game_character_displayname_Ivern":        "",
				"game_character_displayname_Janna":        "",
				"game_character_displayname_JarvanIV":     "",
				"game_character_displayname_Jax":          "",
				"game_character_displayname_Jayce":        "",
				"game_character_displayname_Jhin":         "",
				"game_character_displayname_Jinx":         "",
				"game_character_displayname_Kaisa":        "",
				"game_character_displayname_Kalista":      "",
				"game_character_displayname_Karma":        "",
				"game_character_displayname_Karthus":      "",
				"game_character_displayname_Kassadin":     "",
				"game_character_displayname_Katarina":     "",
				"game_character_displayname_Kayle":        "",
				"game_character_displayname_Kayn":         "",
				"game_character_displayname_Kennen":       "",
				"game_character_displayname_Khazix":       "",
				"game_character_displayname_Kindred":      "",
				"game_character_displayname_Kled":         "",
				"game_character_displayname_KogMaw":       "",
				"game_character_displayname_Leblanc":      "",
				"game_character_displayname_LeeSin":       "",
				"game_character_displayname_Leona":        "",
				"game_character_displayname_Lissandra":    "",
				"game_character_displayname_Lucian":       "",
				"game_character_displayname_Lulu":         "",
				"game_character_displayname_Lux":          "",
				"game_character_displayname_Malphite":     "",
				"game_character_displayname_Malzahar":     "",
				"game_character_displayname_Maokai":       "",
				"game_character_displayname_MasterYi":     "",
				"game_character_displayname_MissFortune":  "",
				"game_character_displayname_MonkeyKing":   "",
				"game_character_displayname_Mordekaiser":  "",
				"game_character_displayname_Morgana":      "",
				"game_character_displayname_Nami":         "",
				"game_character_displayname_Nasus":        "",
				"game_character_displayname_Nautilus":     "",
				"game_character_displayname_Neeko":        "",
				"game_character_displayname_Nidalee":      "",
				"game_character_displayname_Nocturne":     "",
				"game_character_displayname_Nunu":         "",
				"game_character_displayname_Olaf":         "",
				"game_character_displayname_Orianna":      "",
				"game_character_displayname_Ornn":         "",
				"game_character_displayname_Pantheon":     "",
				"game_character_displayname_Poppy":        "",
				"game_character_displayname_Pyke":         "",
				"game_character_displayname_Qiyana":       "",
				"game_character_displayname_Quinn":        "",
				"game_character_displayname_Rakan":        "",
				"game_character_displayname_Rammus":       "",
				"game_character_displayname_RekSai":       "",
				"game_character_displayname_Renekton":     "",
				"game_character_displayname_Rengar":       "",
				"game_character_displayname_Riven":        "",
				"game_character_displayname_Rumble":       "",
				"game_character_displayname_Ryze":         "",
				"game_character_displayname_Sejuani":      "",
				"game_character_displayname_Shaco":        "",
				"game_character_displayname_Shen":         "",
				"game_character_displayname_Shyvana":      "",
				"game_character_displayname_Singed":       "",
				"game_character_displayname_Sion":         "",
				"game_character_displayname_Sivir":        "",
				"game_character_displayname_Skarner":      "",
				"game_character_displayname_Sona":         "",
				"game_character_displayname_Soraka":       "",
				"game_character_displayname_Swain":        "",
				"game_character_displayname_Sylas":        "",
				"game_character_displayname_Syndra":       "",
				"game_character_displayname_TahmKench":    "",
				"game_character_displayname_Taliyah":      "",
				"game_character_displayname_Talon":        "",
				"game_character_displayname_Taric":        "",
				"game_character_displayname_Teemo":        "",
				"game_character_displayname_Thresh":       "",
				"game_character_displayname_Tristana":     "",
				"game_character_displayname_Trundle":      "",
				"game_character_displayname_Tryndamere":   "",
				"game_character_displayname_TwistedFate":  "",
				"game_character_displayname_Twitch":       "",
				"game_character_displayname_Udyr":         "",
				"game_character_displayname_Urgot":        "",
				"game_character_displayname_Varus":        "",
				"game_character_displayname_Vayne":        "",
				"game_character_displayname_Veigar":       "",
				"game_character_displayname_Velkoz":       "",
				"game_character_displayname_Vi":           "",
				"game_character_displayname_Viktor":       "",
				"game_character_displayname_Vladimir":     "",
				"game_character_displayname_Volibear":     "",
				"game_character_displayname_Warwick":      "",
				"game_character_displayname_Xayah":        "",
				"game_character_displayname_Xerath":       "",
				"game_character_displayname_XinZhao":      "",
				"game_character_displayname_Yasuo":        "",
				"game_character_displayname_Yorick":       "",
				"game_character_displayname_Yuumi":        "",
				"game_character_displayname_Zac":          "",
			},
		},
		{
			path: []PathNode{
				{key: "gameData"},
				{key: "gameMode"},
			},
			valueTags: map[string]string{
				"PRACTICETOOL": "c6045372-7714-442d-9a53-0c03bf73d581",
			},
		},
	}
}

func getDynamicFieldValues() []ValuePath {
	return []ValuePath{
		{
			path: []PathNode{
				{key: "activePlayer"},
				{key: "summonerName"},
			},
			valueName: "playerName",
		},
	}
}

type ValuePath struct {
	path      []PathNode
	valueName string
}

type FieldPath struct {
	path      []PathNode
	valueTags map[string]string
}

type PathNode struct {
	key   string
	isRef bool
}
