package processor

import (
	"errors"
	"testing"

	"code.justin.tv/devhub/mdaas-tags-translator/internal/clients/redis"
	rmock "code.justin.tv/devhub/mdaas-tags-translator/internal/clients/redis/mocks"
	"code.justin.tv/devhub/mdaas-tags-translator/models"
	"github.com/sirupsen/logrus"
	"github.com/sirupsen/logrus/hooks/test"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

const (
	invalidJSON                = "invalidjson"
	missingReferenceValuesJSON = `{
		"activePlayer": {
			"WrongKey": "uhh"
		},
		"allPlayers": {
			"randomCharacter": {
				"rawChampionName": "game_character_displayname_Classic"
			}
		},
		"gameData": {
			"gameMode": "PRACTICETOOL"
		}
	}`
	missingValuesJSON = `{
		"activePlayer": {
			"summonerName": "randomCharacter"
		}
	}`
	validJSON = `{
		"activePlayer": {
			"summonerName": "randomCharacter"
		},
		"allPlayers": {
			"randomCharacter": {
				"rawChampionName": "game_character_displayname_Classic"
			}
		},
		"gameData": {
			"gameMode": "PRACTICETOOL"
		}
	}`
	noTagJSON = `{
		"activePlayer": {
			"summonerName": "randomCharacter"
		},
		"allPlayers": {
			"randomCharacter": {
				"rawChampionName": "noTag"
			}
		},
		"gameData": {
			"gameMode": "PRACTICETOOL"
		}
	}`
	TestBroadcaster1 = "TestBroadcaster1"
	TestBroadcaster2 = "TestBroadcaster2"
)

func TestProcess(t *testing.T) {
	r := &rmock.RedisClient{}

	logger, hook := test.NewNullLogger()
	p := NewTestProcessor(r, logger)

	t.Run("Logs error if json is malformed", func(t *testing.T) {
		err := p.Process(models.GameData{
			Data: invalidJSON,
		})

		assert.Error(t, err)
		assert.Equal(t, 0, len(hook.Entries))
		hook.Reset()
	})

	t.Run("Logs error if dynamic value is not found at path", func(t *testing.T) {
		err := p.Process(models.GameData{
			Data: missingReferenceValuesJSON,
		})

		assert.NoError(t, err)
		assert.Equal(t, 3, len(hook.Entries))
		assert.Equal(t, logrus.ErrorLevel, hook.Entries[0].Level)
		assert.Equal(t, "Not found", hook.Entries[0].Message)
		assert.Equal(t, logrus.WarnLevel, hook.Entries[1].Level)
		assert.Equal(t, "Path can not be resolved. Pathnode reference key `playerName` is not referencing an existing value",
			hook.Entries[1].Message)
		assert.Equal(t, logrus.ErrorLevel, hook.LastEntry().Level)
		assert.Equal(t, "Not found", hook.LastEntry().Message)
		hook.Reset()
	})

	t.Run("Logs error if value is not found at path", func(t *testing.T) {
		r.On("Get", mock.Anything, TestBroadcaster1).Return("null", true).Once()

		err := p.Process(models.GameData{
			BroacastersID: []string{TestBroadcaster1},
			Data:          missingValuesJSON,
		})

		assert.NoError(t, err)
		assert.Equal(t, 2, len(hook.Entries))
		assert.Equal(t, logrus.ErrorLevel, hook.LastEntry().Level)
		assert.Equal(t, "Not found", hook.LastEntry().Message)
		hook.Reset()
	})

	t.Run("Logs error when Redis set request fails", func(t *testing.T) {
		r.On("Get", mock.Anything, TestBroadcaster1).Return("null", false).Once()
		r.On("Set", mock.Anything, TestBroadcaster1, `["c122215b-a029-4a32-b812-c263c0268310","c6045372-7714-442d-9a53-0c03bf73d581"]`).Return(errors.New("failed")).Once()

		err := p.Process(models.GameData{
			BroacastersID: []string{TestBroadcaster1},
			Data:          validJSON,
		})

		assert.NoError(t, err)
		assert.Equal(t, 2, len(hook.Entries))
		assert.Equal(t, logrus.ErrorLevel, hook.Entries[0].Level)
		assert.Equal(t, logrus.InfoLevel, hook.LastEntry().Level)
		assert.Equal(t, "failed", hook.Entries[0].Message)
		assert.Equal(t, "Updated channel TestBroadcaster1", hook.LastEntry().Message)
		hook.Reset()
	})

	t.Run("Warns when tags aren't set for value", func(t *testing.T) {
		r.On("Get", mock.Anything, TestBroadcaster1).Return("null", false).Once()
		r.On("Set", mock.Anything, TestBroadcaster1, `["c6045372-7714-442d-9a53-0c03bf73d581"]`).Return(nil).Once()

		err := p.Process(models.GameData{
			BroacastersID: []string{TestBroadcaster1},
			Data:          noTagJSON,
		})

		assert.NoError(t, err)
		assert.Equal(t, 2, len(hook.Entries))
		assert.Equal(t, logrus.WarnLevel, hook.Entries[0].Level)
		assert.Equal(t, logrus.InfoLevel, hook.LastEntry().Level)
		assert.Equal(t, `Missing tag for "noTag"`, hook.Entries[0].Message)
		assert.Equal(t, "Updated channel TestBroadcaster1", hook.LastEntry().Message)
		hook.Reset()
	})

	t.Run("Rules successfully match", func(t *testing.T) {
		r.On("Get", mock.Anything, TestBroadcaster1).Return("null", false).Once()
		r.On("Set", mock.Anything, TestBroadcaster1, `["c122215b-a029-4a32-b812-c263c0268310","c6045372-7714-442d-9a53-0c03bf73d581"]`).Return(nil).Once()

		err := p.Process(models.GameData{
			BroacastersID: []string{TestBroadcaster1},
			Data:          validJSON,
		})

		assert.NoError(t, err)
		assert.Equal(t, 1, len(hook.Entries))
		assert.Equal(t, logrus.InfoLevel, hook.LastEntry().Level)
		assert.Equal(t, "Updated channel TestBroadcaster1", hook.LastEntry().Message)
		hook.Reset()
	})

	t.Run("Handles multiple broadcasters", func(t *testing.T) {
		r.On("Get", mock.Anything, TestBroadcaster1).Return(`["c122215b-a029-4a32-b812-c263c0268310","c6045372-7714-442d-9a53-0c03bf73d581"]`, true).Once()
		r.On("Get", mock.Anything, TestBroadcaster2).Return(`["c122215b-a029-4a32-b812-c263c0268310","c6045372-7714-442d-9a53-0c03bf73d581"]`, true).Once()

		err := p.Process(models.GameData{
			BroacastersID: []string{TestBroadcaster1, TestBroadcaster2},
			Data:          validJSON,
		})

		assert.NoError(t, err)
		assert.Equal(t, 0, len(hook.Entries))
		hook.Reset()
	})

	r.AssertExpectations(t)
}

func TestPathToString(t *testing.T) {
	t.Run("Returns empty list of strings if there are no pathnodes", func(t *testing.T) {
		expected := []string(nil)

		actual, err := pathToString([]PathNode{}, nil)
		assert.NoError(t, err)
		assert.Equal(t, expected, actual)

		actual, err = pathToString(nil, nil)
		assert.NoError(t, err)
		assert.Equal(t, expected, actual)

		actual, err = pathToString(nil, map[string]string{
			"test": "test",
		})
		assert.NoError(t, err)
		assert.Equal(t, expected, actual)
	})

	t.Run("Converts pathnodes into list of strings", func(t *testing.T) {
		expected := []string{"allPlayers", "playerName", "rawChampionName"}

		actual, err := pathToString([]PathNode{
			{key: "allPlayers"},
			{key: "playerName"},
			{key: "rawChampionName"},
		}, nil)
		assert.NoError(t, err)
		assert.Equal(t, expected, actual)

		actual, err = pathToString([]PathNode{
			{key: "allPlayers"},
			{key: "playerName"},
			{key: "rawChampionName"},
		}, map[string]string{
			"playerName": "unwantedstring",
		})
		assert.NoError(t, err)
		assert.Equal(t, expected, actual)
	})

	t.Run("Uses value from values map if node has isRef set to true", func(t *testing.T) {
		expected := []string{"allPlayers", "valueReplacement", "rawChampionName"}

		actual, err := pathToString([]PathNode{
			{key: "allPlayers"},
			{key: "playerName", isRef: true},
			{key: "rawChampionName"},
		}, map[string]string{
			"playerName": "valueReplacement",
		})
		assert.NoError(t, err)
		assert.Equal(t, expected, actual)
	})

	t.Run("Returns empty string if node has isRef but key is not found in map", func(t *testing.T) {
		actual, err := pathToString([]PathNode{
			{key: "allPlayers"},
			{key: "playerName", isRef: true},
			{key: "rawChampionName"},
		}, map[string]string{})
		assert.Error(t, err)
		assert.Equal(t, []string{}, actual)
	})
}

func NewTestProcessor(r redis.RedisClient, logger *logrus.Logger) *ProcessorImpl {
	return &ProcessorImpl{
		Logger: logger,
		Redis:  r,
	}
}
