package naiive

import (
	"context"
	"fmt"
	"go/ast"
	"go/token"
	"io/ioutil"
	"path/filepath"
	"regexp"
	"runtime"
	"strings"

	"golang.org/x/sync/errgroup"
	"golang.org/x/tools/go/analysis"
)

//MinSeverity(MediumSeverity).MinConfidence(MediumConfidence).Analyzer()

// CombineRegex takes a set of regular expressions and converts them to a single regular expression
// as a performance optimization. Each input regular expression is given a capture group, and,
// as such regexes are not combinable if they contain capture groups.
func CombineRegex(prefix string, r ...*regexp.Regexp) (combined *regexp.Regexp, err error) {
	var regexParts []string
	for _, re := range r {
		if re == nil {
			panic(fmt.Sprintf("a regex is nil! %+v", r))
		}

		if sub := re.NumSubexp(); sub != 0 {
			err = fmt.Errorf("cannot combine regex %s, which has %d subexpression(s) and cannot have any", re, sub)
			return
		}

		regexParts = append(regexParts, "("+re.String()+")")
	}

	info("regex parts", regexParts)

	return regexp.Compile(prefix + "(?:" + strings.Join(regexParts, "|") + ")")
}

type Tests []Test

var debug bool

func info(i ...interface{}) {
	if debug {
		fmt.Println(i...)
	}
}
func infof(format string, i ...interface{}) {
	if debug {
		fmt.Printf(format, i...)
	}
}

func (t Tests) Analyzer() *analysis.Analyzer {
	analyzer := &analysis.Analyzer{
		Name: "naiive",
		Doc:  "scan dependencies for security issues via regex",
		Run:  t.run,
	}

	analyzer.Flags.BoolVar(&debug, "n-debug", false, "spew debug info")

	return analyzer
}

func (t Tests) run(pass *analysis.Pass) (interface{}, error) {
	return t.Run(pass, context.Background())
}

func (t Tests) Run(pass *analysis.Pass, ctx context.Context) (out interface{}, err error) {
	if len(t) == 0 {
		panic("no tests!")
	}

	re, err := t.combineRegex()
	if err != nil {
		return
	}

	return (run{Regex: re, Tests: t, pass: pass}).Run(ctx)
}

func (t Tests) combineRegex() (re *regexp.Regexp, err error) {
	info("tests", t)
	var res = make([]*regexp.Regexp, len(t))
	for i, v := range t {
		res[i] = v.Regexp
	}

	info("regexps", res)

	if re, err = CombineRegex("", res...); err != nil {
		return
	}

	return
}

type run struct {
	Regex *regexp.Regexp
	Tests Tests
	pass  *analysis.Pass
}

func (r run) Run(ctx context.Context) (out interface{}, err error) {
	group, ctx := errgroup.WithContext(ctx)

	var chFiles = make(chan *ast.File)

	for i := 0; i < runtime.NumCPU(); i++ {
		group.Go(func() (err error) {
			for file := range chFiles {
				if err = r.CheckFile(file); err != nil {
					return
				}

			}

			return
		})
	}

	for _, file := range r.pass.Files {
		chFiles <- file
	}
	close(chFiles)

	return nil, group.Wait()
}

func (r run) CheckFile(f *ast.File) (err error) {
	var file = r.pass.Fset.File(f.Pos())
	info("checking file", file.Name())

	// ignore go test files
	if strings.HasSuffix(file.Name(), "_test.go") {
		return
	}

	// ignore cgo files
	if filepath.Base(file.Name()) == "C" {
		return
	}

	bt, err := ioutil.ReadFile(file.Name())
	if err != nil {
		return
	}

	matches := r.Regex.Copy().FindAllSubmatchIndex(bt, -1)
	info("re", r.Regex)
	info("searched", file.Name(), len(matches), "matches")
	for _, matchGroup := range matches {
		matchGroup = matchGroup[2:] // strip off 'whole regex match'
		for i := 0; i < len(matchGroup)/2; i++ {
			indexes := matchGroup[i*2 : i*2+2]
			start, end := indexes[0], indexes[1]
			info("checking start...")
			if start == -1 { // no match
				continue
			}

			test := r.Tests[i]

			info("passing to Report()...")
			// NB: Pos is basically a byte offset.
			// in v1, we just called f.Pos(offset), which
			// gave us the offset in the file -- but that's
			//a token.File, not ast.File method, so here we
			// just add the beginning of the file offset
			// to the offset we got in the file.
			r.Report([2]token.Pos{
				file.Pos(start),
				file.Pos(end),
			}, test)
		}
	}

	return
}

func (r run) Report(pos [2]token.Pos, test Test) {
	info("reporting !!")
	r.pass.Report(analysis.Diagnostic{
		Pos:     pos[0],
		End:     pos[1],
		Message: test.Name,
	})
}
