import { Address, AuthData, LogLevel, Register, Registry, SetLevel, Writer } from 'extension-messaging-prototype';
import { CycleListener } from './cycle_listener';
import { debugInit } from './debug_init';

const host = 'wss://connect.prod.eml.twitch.a2z.com/';

const msg = new Uint8Array([99, 121, 99, 108, 101]).buffer; // 'cycle'

let reg: Registry | null = null;
let writer: Writer | null = null;
let addr: Address | null = null;
let level = LogLevel.Error;
let authData: AuthData = {
  clientId: '',
  token: '',
};

const cycle = new CycleListener();
cycle.onCycle = onCycle;

function onCycle() {
  const color = document.getElementById('color');
  if (color) {
    color.style.background = cycle.color;
  }
}

async function initialize(channel: string): Promise<void> {
  await subscribe(channel);
  const button = document.getElementById('cycle');
  if (button) {
    button.onclick = () => {
      if (addr && reg) {
        if (writer === null) {
          writer = reg.writer(addr);
        }
        writer.send(msg, false);
      }
    };
    button.removeAttribute('disabled');
  }
}

async function subscribe(channel: string): Promise<void> {
  const newAddr = Address.forNamespace('ext')
    .withVersion(1)
    .withFilter('a', '*')
    .withFilter('e', authData.clientId)
    .withFilter('c', channel)
    .build();

  if (addr) {
    if (newAddr.key === addr.key) {
      return;
    }
    if (reg) {
      if (writer) {
        const w = writer;
        writer = null;
        await w.close();
      }
      await reg.reader(addr).leave(cycle);
    }
  }
  addr = newAddr;
  if (reg) {
    await reg.reader(addr).join(cycle);
  }
}

document.addEventListener('DOMContentLoaded', () => {
  if (!window.Twitch || !window.Twitch.ext) {
    throw new Error('Unable to locate the Twitch helper library');
  }

  const ext = window.Twitch.ext;
  const log = SetLevel(() => level, ext.rig.log);
  reg = Register(host, () => authData, log);
  ext.onAuthorized((auth) => {
    authData = auth;
    if (reg) {
      reg.refreshAuth();
    }
    initialize(auth.channelId);
  });
  onCycle();

  // for local debugging only
  debugInit(
    (l: LogLevel) => { level = l; },
    (a: AuthData) => { authData = a; },
    initialize,
    log,
  );
});
