import { Address, AuthData, LogLevel, Register, Registry, SetLevel } from 'extension-messaging-prototype';
import { ColorListener } from './color_listener';
import { debugInit } from './debug_init';

const host = 'wss://connect.prod.eml.twitch.a2z.com/';
const msg = new Uint8Array([99, 121, 99, 108, 101]).buffer; // 'cycle'

let reg: Registry | null = null;
let colorTopic: Address | null = null;
let cycleTopic: Address | null = null;
let level: LogLevel = LogLevel.Info;
let authData: AuthData = {
  clientId: '',
  token: '',
};

const listener = new ColorListener();
listener.onSet = onSet;

function onSet() {
  const color = document.getElementById('color');
  if (color) {
    color.style.background = listener.color;
  }
}

async function initialize(channelId: string): Promise<void> {
  await subscribe(channelId);
  const button = document.getElementById('cycle');
  if (button) {
    button.onclick = () => {
      if (cycleTopic && reg) {
        reg.writer(cycleTopic).send(msg, false);
      }
    };
    button.removeAttribute('disabled');
  }
}

function address(channelId: string, topic: string): Address {
  return Address.forNamespace('ext')
    .withVersion(1)
    .withFilter('c', channelId)
    .withFilter('a', '*')
    .withFilter('e', authData.clientId)
    .withFilter('x', topic).build();
}

async function subscribe(channelId: string): Promise<void> {
  const newColorTopic = address(channelId, 'color');
  if (colorTopic) {
    if (newColorTopic.key === colorTopic.key) {
      return;
    }
    if (reg) {
      await reg.reader(colorTopic).leave(listener);
    }
  }
  colorTopic = newColorTopic;
  cycleTopic = address(channelId, 'cycle');
  if (reg) {
    await reg.reader(colorTopic).join(listener);
  }
}

document.addEventListener('DOMContentLoaded', () => {
  if (!window.Twitch || !window.Twitch.ext) {
    throw new Error('Unable to locate the Twitch helper library');
  }

  const ext = window.Twitch.ext;
  const log = SetLevel(() => level, ext.rig.log);
  reg = Register(host, () => authData, log);
  ext.onAuthorized((auth) => { authData = auth; initialize(auth.channelId); });
  onSet();

  // for local debugging only
  debugInit(
    (l: number) => { level = l; },
    (a: AuthData) => { authData = a; },
    initialize,
    log,
  );
});
