// tslint:disable:no-console
import * as Color from 'color';
import { Address, AuthData, Listener, LogLevel, Position, Register, Segment, SetLevel, Source } from 'extension-messaging-prototype';
import * as jwt from 'jsonwebtoken';

const durationSeconds = 60; // TTL on auth token
const host = 'wss://connect.prod.eml.twitch.a2z.com/';
const reg = Register(host, createAuth, SetLevel(LogLevel.Debug, console.log));
let clientId = '';
let secret = '';
let userId = '';

process.argv.forEach((val) => {
  if (val.startsWith('-')) {
    const segments = val.split('=');
    if (segments.length === 2) {
      switch (segments[0]) {
        case '-user-id': userId = segments[1]; break;
        case '-client-id': clientId = segments[1]; break;
        case '-extension-secret': secret = segments[1]; break;
      }
    }
  }
});

function createAuth(): AuthData {
  const exp = Math.floor((Date.now() / 1000) + durationSeconds);
  const token = jwt.sign({
      exp,
      pubsub_perms: {
        listen: ['*'],
        send: ['*'],
      },
      role: 'external',
      channel_id: '0',
      user_id: userId,
    },
    Buffer.from(secret, 'base64'),
    { algorithm: 'HS256' },
  );
  return {
    clientId,
    token,
  };
}

class CycleListener implements Listener {
  private readonly pos = new Map<string, Position>();

  public currentPosition(addr: Address): Position | undefined {
    return this.pos.get(addr.key);
  }

  public onDataLost(addr: Address, src: Source, at: Segment): void {
    this.update(addr, src, at.end);
  }

  // use the segment end for each topic to determine color, data isn't relevant
  public onDataReceived(addr: Address, src: Source, at: Segment): void {
    this.update(addr, src, at.end);
  }

  public onStreamClosed(addr: Address): void {
    this.pos.delete(addr.key);
  }

  private update(addr: Address, src: Source, end: number) {
    const pos = this.currentPosition(addr);
    if (pos && pos.src.id === src.id && pos.at >= end) {
      return;
    }
    this.pos.set(addr.key, { src, at: end });
    // convert incoming cycle address to outgoing color address with same params
    const dest = Address.parse(addr.key.replace('x=cycle', 'x=color'));
    const color = new Color('#6441A4').rotate(end * 50);
    console.log(addr, color);
    const msg = new Uint8Array(3);
    msg[0] = color.red();
    msg[1] = color.green();
    msg[2] = color.blue();
    reg.writer(dest).send(msg.buffer, false);
  }
}

reg.reader(
  // NOTE: this address handles cycle messages from *all* channels because it
  // doesn't specify one
  Address.forNamespace('ext')
    .withVersion(1)
    .withFilter('e', clientId)
    .withFilter('a', '*')
    .withFilter('x', 'cycle')
    .build(),
).join(new CycleListener());

console.log('Initialized');
