import * as iam from '@aws-cdk/aws-iam';
import * as cdk from '@aws-cdk/core';
import * as dynamodb from '@aws-cdk/aws-dynamodb';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as ecs from '@aws-cdk/aws-ecs';
import * as path from 'path';
import * as sqs from '@aws-cdk/aws-sqs';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { PROD_ACCOUNT_ID, LIVELINE_HOST, StrategyConfig } from './consts';

interface ProcessorStackProps extends cdk.StackProps {
  strategy: StrategyConfig;
  channelsQueue: sqs.Queue;
  channelsTable: dynamodb.Table;
  oauthTokenSecret: secretsmanager.Secret;
  vpc: ec2.Vpc;
}

export class ProcessorStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: ProcessorStackProps) {
    super(scope, id, props);

    const prod = props.env!.account === PROD_ACCOUNT_ID;

    // Cluster
    const cluster = new ecs.Cluster(this, 'Cluster', {
      vpc: props.vpc,
    });
    cdk.Tag.add(cluster, 'ecs.Cluster', props.strategy.name + 'Processor');

    // ASG
    const asg = cluster.addCapacity('Asg', {
      instanceType: new ec2.InstanceType('m5.large'),
      minCapacity: prod ? props.strategy.minClusterCapacity : 1,
      maxCapacity: prod ? props.strategy.maxClusterCapacity : 1,
      vpcSubnets: {
        // We explicitly place this cluster locked down in the public subnet for cost.
        // See https://jira.twitch.com/browse/SEC-4518 for details and security approval.
        subnetType: ec2.SubnetType.PUBLIC,
      },
    });

    asg.scaleToTrackMetric('CpuReservation', {
      metric: cluster.metricCpuReservation(),
      targetValue: 95,
      cooldown: cdk.Duration.seconds(60),
      estimatedInstanceWarmup: cdk.Duration.seconds(60),
    });

    asg.scaleToTrackMetric('MemoryReservation', {
      metric: cluster.metricMemoryReservation(),
      targetValue: 95,
      cooldown: cdk.Duration.seconds(60),
      estimatedInstanceWarmup: cdk.Duration.seconds(60),
    });

    // Task
    const taskDefinition = new ecs.Ec2TaskDefinition(this, 'TaskDef');
    cdk.Tag.add(taskDefinition, 'ecs.TaskDefinition', props.strategy.name + 'Processor');

    taskDefinition.addContainer('processor', {
      environment: {
        AWS_DEFAULT_REGION: props.env!.region!,
        CHANNELS_QUEUE: props.channelsQueue.queueUrl,
        CHANNELS_TABLE: props.channelsTable.tableName,
        DEV: prod ? '' : '1',
        LIVELINE_HOST,
        OAUTH_TOKEN_SECRET: props.oauthTokenSecret.secretArn,
        CATEGORY_ID: props.strategy.category_id.toString(),
        STRATEGY: props.strategy.name.toLowerCase(),
      },
      image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../../build/processor')),
      logging: new ecs.AwsLogDriver({
        streamPrefix: 'processor',
        logRetention: 30,
      }),
      memoryLimitMiB: props.strategy.taskMemory,
    });

    props.channelsQueue.grantConsumeMessages(taskDefinition.taskRole);
    props.channelsTable.grantReadWriteData(taskDefinition.taskRole);
    props.oauthTokenSecret.grantRead(taskDefinition.taskRole);

    taskDefinition.addToTaskRolePolicy(
      new iam.PolicyStatement({
        actions: ['cloudwatch:PutMetricData'],
        effect: iam.Effect.ALLOW,
        resources: ['*'],
      })
    );

    // Service
    const service = new ecs.Ec2Service(this, 'Service', {
      cluster,
      taskDefinition,
      desiredCount: prod ? props.strategy.taskCount : 1,
    });
    cdk.Tag.add(service, 'ecs.Service', props.strategy.name + 'Processor');
  }
}
