import * as cdk from '@aws-cdk/core';
import * as dynamodb from '@aws-cdk/aws-dynamodb';
import * as ec2 from '@aws-cdk/aws-ec2';
import * as events from '@aws-cdk/aws-events';
import * as lambda from '@aws-cdk/aws-lambda';
import * as path from 'path';
import * as sqs from '@aws-cdk/aws-sqs';
import * as targets from '@aws-cdk/aws-events-targets';
import { PROD_ACCOUNT_ID, LIVELINE_HOST, StrategyConfig } from './consts';

interface SchedulerStackProps extends cdk.StackProps {
  strategy: StrategyConfig;
  channelsQueue: sqs.Queue;
  channelsTable: dynamodb.Table;
  vpc: ec2.Vpc;
}

export class SchedulerStack extends cdk.Stack {
  constructor(scope: cdk.Construct, id: string, props: SchedulerStackProps) {
    super(scope, id, props);

    const prod = props.env!.account === PROD_ACCOUNT_ID;

    // Function
    const func = new lambda.Function(this, 'Function', {
      code: lambda.Code.fromAsset(path.join(__dirname, '../../build/scheduler')),
      handler: 'scheduler.main',
      logRetention: 30,
      runtime: lambda.Runtime.PYTHON_3_7,
      timeout: cdk.Duration.seconds(60),
      vpc: props.vpc,
    });

    // Environment
    func.addEnvironment('CATEGORY_ID', props.strategy.category_id.toString());
    func.addEnvironment('CHANNELS_QUEUE', props.channelsQueue.queueUrl);
    func.addEnvironment('CHANNELS_TABLE', props.channelsTable.tableName);
    func.addEnvironment('LIVELINE_HOST', LIVELINE_HOST);
    if (!prod) {
      func.addEnvironment('DEV', '1');
    }

    // Permissions
    props.channelsQueue.grantSendMessages(func);
    props.channelsTable.grantReadWriteData(func);

    // Minutely event
    const lambdaRule = new events.Rule(this, 'Schedule', {
      schedule: events.Schedule.expression('rate(1 minute)'),
    });
    lambdaRule.addTarget(new targets.LambdaFunction(func));
  }
}
