#!/usr/bin/env python3

import logging
import os
import signal
import sys
from typing import Optional
from importlib import import_module

import boto3
from pythonjsonlogger import jsonlogger

from .processor import Processor

DEV = os.environ.get("DEV") == "1"
QUEUE = os.environ["CHANNELS_QUEUE"]
TABLE = os.environ["CHANNELS_TABLE"]
OAUTH_TOKEN_SECRET = os.environ["OAUTH_TOKEN_SECRET"]
STRATEGY = os.environ["STRATEGY"].lower()
CATEGORY_ID = int(os.environ["CATEGORY_ID"])

logger = logging.getLogger("processor")


def shutdown_signal_handler(signum: int, frame: Optional[object]):
    logger.info(f"Received signal {signum}, exiting")
    sys.exit(0)


def main():
    logHandler = logging.StreamHandler()
    logHandler.setFormatter(
        jsonlogger.JsonFormatter("(message)(levelname)(name)(asctime)")
    )
    logging.root.addHandler(logHandler)
    logging.root.setLevel(logging.INFO)

    logger.info("Starting", extra={"strategy": STRATEGY, "category_id": CATEGORY_ID})

    signal.signal(signal.SIGTERM, shutdown_signal_handler)
    signal.signal(signal.SIGINT, shutdown_signal_handler)

    logger.info("Getting OAuth token from Secrets Manager")
    secrets_manager = boto3.client("secretsmanager")
    oauth_token = secrets_manager.get_secret_value(SecretId=OAUTH_TOKEN_SECRET)

    logger.info("Loading strategy")
    strategy = import_module(f".strategies.{STRATEGY}", __package__).Strategy

    logger.info("Creating Processor")
    Processor(
        strategy_name=STRATEGY,
        strategy=strategy,
        category_id=CATEGORY_ID,
        channels_queue=QUEUE,
        channels_table=TABLE,
        oauth_token=oauth_token["SecretString"],
        debug=DEV,
    ).run()


if __name__ == "__main__":
    main()
