import json
import logging
import math
import time
from threading import Thread
from datetime import datetime

import boto3
import requests
import strevr

from .liveline import get_channel_info

# Minimum number of viewers to process a channel.
MIN_CCU = 3
# Maximum time in seconds between strevr results before we stop processing the channel.
STREVR_LAST_TIME_MAX = 300
# Minimum confidence before considering a match successful
MIN_CONFIDENCE = 0.94

logger = logging.getLogger(__name__)


class Processor:
    def __init__(
        self,
        *,
        strategy_name: str,
        strategy: object,
        category_id: int,
        channels_queue: str,
        channels_table: str,
        oauth_token: str,
        debug=False,
    ):
        self._metrics = []
        self._last_metrics_flush = time.time()
        self._category_id = category_id
        self._strategy_name = strategy_name

        logger.debug("Creating Boto3 clients and resources")
        self._debug = debug
        self._cloudwatch = boto3.client("cloudwatch")
        self._queue = boto3.resource("sqs").Queue(channels_queue)
        self._dynamodb = boto3.resource("dynamodb")
        self._table = self._dynamodb.Table(channels_table)

        logger.info(
            "Initializing Strevr", extra={"strategy": self._strategy_name,},
        )
        self._strevr = strevr.Strevr(
            strategy,
            minimum_confidence=MIN_CONFIDENCE,
            is_debugging=debug,
            e2_access_token=oauth_token,
        )

    def _track_metric(self, name: str, value=1.0):
        """ Tracks a simple metric in CloudWatch. """
        logger.debug(f"Tracking metric: {name}={value} ")
        self._metrics.append(
            {
                "MetricName": name,
                "Value": value,
                "Timestamp": datetime.utcnow(),
                "Dimensions": [{"Name": "strategy", "Value": self._strategy_name,}],
            }
        )

        if time.time() - self._last_metrics_flush >= 60:
            self._flush_metrics()

    def _flush_metrics(self):
        """ Sends all tracked metrics to CloudWatch. """
        logger.debug("Flushing metrics")
        self._cloudwatch.put_metric_data(
            MetricData=self._metrics, Namespace="metavision/processor",
        )
        self._last_metrics_flush = time.time()
        self._metrics = []

    def _validate_channel(self, channel: dict) -> bool:
        """ Checks whether a channel is eligible to be processed. """
        logger.debug(
            "Validating channel", extra={"channel": channel["name"]},
        )

        streams = get_channel_info(channel)["streams"]

        if len(streams) == 0:
            logger.info(
                "Channel is offline", extra={"channel": channel["name"]},
            )
            self._track_metric("channel_offline")
            return False

        category_id = streams[0].get("channel_data", {}).get("category_id", 0)
        if category_id != self._category_id:
            logger.info(
                "Invalid channel category.",
                extra={
                    "channel": channel["name"],
                    "category_id": category_id,
                    "expected_category_id": self._category_id,
                },
            )
            self._track_metric("invalid_channel_category")
            return False

        ccu = int(streams[0]["viewcount_data"].get("viewcount", 0))
        if ccu < MIN_CCU:
            logger.info(
                "Channel CCU too low",
                extra={"channel": channel["name"], "ccu": ccu, "min_ccu": MIN_CCU},
            )
            self._track_metric("channel_ccu_too_low")
            return False

        return True

    def run(self):
        """ Calls process_channel in a loop, ensuring any unhandled errors do not stop the processing. """
        while True:
            try:
                self.process_channel()
            except Exception as e:
                logger.exception("Unhandled error", extra={"exc": type(e).__name__})
                self._track_metric("unhandled_error")
            finally:
                self._flush_metrics()
                time.sleep(1)

    def process_channel(self):
        """ Waits for a channel and processes it. """
        strevr_processor = None
        strevr_thread = None
        try:
            logger.info("Waiting for channel to process")
            self._track_metric("waiting")

            messages = self._queue.receive_messages(WaitTimeSeconds=20)
            if len(messages) == 0:
                logger.debug("No messages received")
                return

            channel = json.loads(messages[0].body)

            logger.info(
                "Received channel", extra={"channel": channel["name"]},
            )
            self._track_metric("processing_started")

            now = math.floor(time.time())
            last_lock_time = now

            logger.debug(
                "Acquiring lock", extra={"channel": channel["name"]},
            )
            try:
                self._table.update_item(
                    Key={"id": channel["id"]},
                    UpdateExpression="SET #time = :now",
                    ConditionExpression="attribute_not_exists(#time) OR #time < :min_time",
                    ExpressionAttributeNames={"#time": "time",},
                    ExpressionAttributeValues={":now": now, ":min_time": now - 60,},
                )
            except self._dynamodb.meta.client.exceptions.ConditionalCheckFailedException:
                logger.info(
                    "Channel is already being processed",
                    extra={"channel": channel["name"]},
                )
                self._track_metric("acquire_lock_failed")
                return
            finally:
                logger.debug("Deleting message")
                result = self._queue.delete_messages(
                    Entries=[
                        {
                            "Id": messages[0].message_id,
                            "ReceiptHandle": messages[0].receipt_handle,
                        }
                    ]
                )

            if not self._validate_channel(channel):
                return

            logger.info(
                "Starting Strevr", extra={"channel": channel["name"]},
            )

            stop_processing = False

            last_match = {
                "detector": "",
                "result": None,
                "confidence": 0,
                "time": now,
            }

            def on_strevr_result(channel_name, detector, result, confidence):
                nonlocal last_match

                if result and confidence >= MIN_CONFIDENCE:
                    if (
                        detector != last_match["detector"]
                        or result != last_match["result"]
                    ):
                        logger.info(
                            "Strevr result",
                            extra={
                                "channel": channel_name,
                                "detector": detector,
                                "result": result,
                                "confidence": round(confidence, 3),
                                "min_confidence": MIN_CONFIDENCE,
                            },
                        )

                    last_match["detector"] = detector
                    last_match["result"] = result
                    last_match["confidence"] = confidence
                    last_match["time"] = time.time()

            strevr_processor = self._strevr.make_processor(
                channel["name"], on_strevr_result
            )

            if strevr_processor is None:
                logger.warning(
                    "Failed to create Strevr processor",
                    extra={"channel": channel["name"]},
                )
                self._track_metric("strevr_start_failed")
                return

            last_active = time.time()
            self._track_metric("active")

            def runner():
                nonlocal stop_processing
                try:
                    strevr_processor.run()
                    logger.info(
                        "Strevr ended for channel", extra={"channel": channel["name"]},
                    )
                    self._track_metric("strevr_ended")
                except Exception as e:
                    logger.exception(
                        "Unhandled Strevr error",
                        extra={"channel": channel["name"], "exc": type(e).__name__,},
                    )
                    self._track_metric("unhandled_strevr_error")
                finally:
                    stop_processing = True

            strevr_thread = Thread(target=runner)
            strevr_thread.start()

            while not stop_processing:
                time.sleep(30)

                now = math.floor(time.time())

                if now - last_active > 60:
                    last_active = now
                    self._track_metric("active")

                if now - last_match["time"] > STREVR_LAST_TIME_MAX:
                    logger.warning(
                        "No result from Strevr within time threshold",
                        extra={
                            "channel": channel["name"],
                            "last_result_time": now - last_match["time"],
                            "last_result_threshold": STREVR_LAST_TIME_MAX,
                        },
                    )
                    self._track_metric("strevr_no_result")
                    stop_processing = True
                    continue

                if not self._validate_channel(channel):
                    stop_processing = True
                    continue

                logger.debug(
                    "Refreshing lock", extra={"channel": channel["name"]},
                )

                try:
                    self._table.update_item(
                        Key={"id": channel["id"]},
                        UpdateExpression="SET #time = :now",
                        ConditionExpression="attribute_not_exists(#time) OR #time = :time",
                        ExpressionAttributeNames={"#time": "time",},
                        ExpressionAttributeValues={
                            ":now": now,
                            ":time": last_lock_time,
                        },
                    )
                except self._dynamodb.meta.client.exceptions.ConditionalCheckFailedException:
                    logger.error(
                        "Refreshing lock failed", extra={"channel": channel["name"]},
                    )
                    self._track_metric("refresh_lock_failed")
                    stop_processing = True
                    continue

                last_lock_time = now
        finally:
            if strevr_processor is not None:
                strevr_processor.stop()

            if strevr_thread is not None:
                strevr_thread.join(10)

                if strevr_thread.is_alive():
                    logger.warning(
                        "Failed to stop Strevr on channel \"{channel['name']}\" within 10 seconds",
                        extra={"channel": channel["name"]},
                    )
                    self._track_metric("strevr_stop_failed")

            self._track_metric("processing_ended")
