#!/usr/bin/env python3

import os
import sys

sys.path.append(os.path.join(os.path.dirname(os.path.realpath(__file__)), "./vendor"))

import json
import logging
import signal
import time
from typing import List, Optional

import boto3
import requests
from more_itertools import chunked

DEV = os.environ.get("DEV") == "1"
QUEUE = os.environ["CHANNELS_QUEUE"]
TABLE = os.environ["CHANNELS_TABLE"]
CATEGORY_ID = os.environ["CATEGORY_ID"]

LIVELINE_HOST = os.environ["LIVELINE_HOST"]
LIVELINE_BASE = f"{LIVELINE_HOST}/twirp/liveline.Liveline"
LIVELINE_GET_FILTERED_CHANNELS = f"{LIVELINE_BASE}/GetFilteredStreams"

MIN_CCU = 3
MAX_CHANNELS = 1000 if not DEV else 10

# Must set level for Lambda logging to work
logging.getLogger().setLevel(logging.INFO)

# Get a named logger for scheduler
logger = logging.getLogger("scheduler")


def get_eligible_channels(max_channels=MAX_CHANNELS, min_ccu=MIN_CCU) -> List[dict]:
    logger.info(
        f"Getting top {max_channels} channels with at least {min_ccu} viewers from liveline for category {CATEGORY_ID}."
    )

    res = requests.post(
        LIVELINE_GET_FILTERED_CHANNELS,
        headers={"Content-Type": "application/json"},
        data=json.dumps(
            {
                "filters": [{"filter_key": 1, "filter_value": [str(CATEGORY_ID)],},],
                "sort_key": 1,
                "order": 1,
                "page_size": max_channels,
            }
        ),
        timeout=10,
    )

    channels = [
        {"id": s["channel_id"], "name": s["channel_name"],}
        for s in res.json()["streams"]
        if int(s["viewcount_data"].get("viewcount", 0)) >= min_ccu
    ]

    logger.info(f"Found {len(channels)} eligible channels.")
    return channels


def remove_active_channels(dynamodb: object, channels: List[str]):
    logger.info("Removing active channels...")

    invalid_channels = set()

    for chunk in chunked(channels, 100):
        res = dynamodb.batch_get_item(
            RequestItems={
                TABLE: {"Keys": [{"id": {"S": channel["id"]}} for channel in chunk],},
            }
        )

        for channel in res["Responses"][TABLE]:
            if int(channel["time"]["N"]) >= time.time() - 60:
                invalid_channels.add(channel["id"]["S"])

    eligible_channels = [
        channel for channel in channels if channel["id"] not in invalid_channels
    ]

    logger.info(f"Removed {len(channels) - len(eligible_channels)} channels.")
    return eligible_channels


def queue_channels(queue: object, channels: List[dict]):
    logger.info(f"Queueing {len(channels)} channels.")

    for chunk in chunked(channels, 10):
        queue.send_messages(
            Entries=[
                {"Id": channel["id"], "MessageBody": json.dumps(channel),}
                for channel in chunk
            ]
        )


def shutdown_signal_handler(signum: int, frame: Optional[object]):
    logger.info(f"Received signal {signum}, exiting.")
    sys.exit(0)


def main(event, context):
    logging.basicConfig(
        format="%(asctime)s %(name)s [%(levelname)s] %(message)s", level=logging.INFO,
    )

    logger.info("Scheduler starting")

    signal.signal(signal.SIGTERM, shutdown_signal_handler)
    signal.signal(signal.SIGINT, shutdown_signal_handler)

    sqs = boto3.resource("sqs")
    queue = sqs.Queue(QUEUE)

    dynamodb = boto3.client("dynamodb")

    channels = get_eligible_channels()
    channels = remove_active_channels(dynamodb, channels)
    queue_channels(queue, channels)


if __name__ == "__main__":
    main(None, None)
