#!/usr/bin/env python3

import json
import logging
import os
from strevr import Strevr
from strevr.utilities import fetch_channel_logins
from data.strategy import game_id, Strategy

def main(channel_login, wants_cpu, is_debugging, e2_access_token, minimum_confidence):
    logging.basicConfig(format="%(asctime)s [%(levelname)s] %(message)s",
                        level=logging.DEBUG if is_debugging else logging.INFO)

    # If the command line does not provide a token, assume there is a file in
    # the data directory containing one.
    if not e2_access_token:
        token_file_path = os.path.join(os.path.dirname(__file__), 'data', 'strategy.json')
        with open(token_file_path) as fin:
            e2_access_token = json.load(fin)['e2_access_token']

    # If the command line does not provide a channel name, select the top one
    # from the API.
    channel_login = fetch_channel_logins(1, game_id, e2_access_token)[0] if channel_login is None else channel_login

    # Use the CPU if requested.
    if wants_cpu:
        from strevr.model import Model
        Model.use_cpu()

    # Start processing, awaiting a ^C key press or stream completion to exit.
    print('Press ^C to exit.')
    strevr = Strevr(Strategy, is_debugging=is_debugging, e2_access_token=e2_access_token, minimum_confidence=minimum_confidence)
    def on_result(channel_login, detector_name, detector_result, detector_confidence):
        print(channel_login, detector_name, detector_result, detector_confidence)
    processor = strevr.make_processor(channel_login, on_result)
    if processor:
        try:
            processor.run()
            print(channel_login, 'has ended')
        except KeyboardInterrupt:
            print('stopping streams')
        except Exception as ex:
            print(ex)
    else:
        print('no stream available for', channel_login)

if __name__ == '__main__':
    # Parse the command line arguments.
    import argparse
    parser = argparse.ArgumentParser(description='Invoke detectors for a given channel.')
    parser.add_argument('--use-cpu', '-c', action='store_true', help='enforce use of the CPU')
    parser.add_argument('--debug', '-d', action='store_true', help='enter debug mode')
    parser.add_argument('--token', '-t', help='E2 access token')
    parser.add_argument('--confidence', '-n', type=float, help='minimum confidence for positive detection')
    parser.add_argument('channel_login', nargs='?', help='the login of the channel to inspect (default top channel)')
    args = parser.parse_args()
    main(args.channel_login, args.use_cpu, args.debug, args.token, args.confidence)
