import abc
import logging
from datetime import timedelta
from time import sleep
from typing import Any, Callable, Optional, Union
try:
    from .frame_provider import FrameProvider
    from .m3u8_parser import ObserverFn, fetch_live_stream, video_api_client_id
    from .utilities import desired_resolution, resize_image
except ImportError:
    from frame_provider import FrameProvider
    from m3u8_parser import ObserverFn, fetch_live_stream, video_api_client_id
    from utilities import desired_resolution, resize_image

CloseFn = Callable[[], None]
ResultFn = Callable[[str, str, Any, float], None]

logger = logging.getLogger(__name__)

class ProcessorBase(abc.ABC):
    @abc.abstractmethod
    def run(self) -> None:
        ...

    @abc.abstractmethod
    def stop(self) -> None:
        ...

class Strevr:
    class Processor(ProcessorBase):
        def __init__(self, channel_login: str, stream: FrameProvider, observer_fn: ObserverFn, close_fn: CloseFn):
            self.__channel_login = channel_login
            self.__close_fn = close_fn
            self.__is_processing = None
            self.__observer_fn = observer_fn
            self.__stream = stream

        def __close(self) -> None:
            self.__close_fn()
            self.__stream.close()

        def __is_processing_(self) -> bool:
            delay = self.__stream.process(self.__observer_fn)
            if delay is None:
                return False
            if delay > 0:
                sleep(delay)
            return True

        def run(self) -> None:
            if self.__is_processing == self.__is_processing_:
                raise AssertionError('unexpected re-entrant invocation')
            self.__is_processing = self.__is_processing_
            try:
                while self.__is_processing():
                    pass
            finally:
                self.__close()

        def stop(self) -> None:
            self.__is_processing = lambda: False

    def __init__(self, Strategy, *, delay: Union[float, int, timedelta]=0, is_debugging: bool=False, **kwargs):
        self.__delay = delay if type(delay) is timedelta else timedelta(seconds=float(delay))

        # Create the Strategy object.
        try:
            from .detector import Detector
        except ImportError:
            from detector import Detector
        self.__is_debugging = is_debugging
        self.__strategy = Strategy(Detector, **kwargs)
        logger.info('loaded strategy')

    def make_processor(self, channel_login: str, on_result: ResultFn=None) -> Optional[ProcessorBase]:
        # Validate the call-back function.
        if on_result is not None and not callable(on_result):
            raise TypeError('on_result is not callable')

        # Get the 480p stream.
        channel_id, stream = fetch_live_stream(video_api_client_id, channel_login, desired_resolution, FrameProvider)
        if not stream:
            logger.warning('no streams available for %s at desired resolution', channel_login)
            return
        logger.info('selected %dx%d stream for %s', *stream.resolution, channel_login)

        # Open the strategy.
        process_fn_, close_fn = self.__strategy.open(channel_login, channel_id, self.__is_debugging, on_result)
        process_fn = process_fn_ if stream.resolution[1] == desired_resolution else lambda f: process_fn_(resize_image(f))

        # Create a function to process frames.
        durations = [timedelta(0), timedelta(0)]
        def process_frames(segment, duration):
            logger.debug(f'm3u8 manifest reports {round(duration.total_seconds() * 1000)} ms segment duration')
            durations[0] += duration
            if durations[0] >= durations[1]:
                # Invoke the strategy on the frames.
                g = iter(segment)
                result = any(map(process_fn, g))
                durations[:] = timedelta(0), self.__delay if result else timedelta(0)
                return True
            else:
                return False

        # Start processing the stream.
        return Strevr.Processor(channel_login, stream, process_frames, close_fn)
