import cv2
import numpy as np
from typing import Any, Tuple, Union
try:
    from .model import Model
    from .utilities import change_image_type, load_parameters
except ImportError:
    from model import Model
    from utilities import change_image_type, load_parameters

class Detector:
    load_parameters = load_parameters
    Model = Model

    def __init__(self, parameters: Union[dict, str], model: Union[Model, str]):
        if type(parameters) is str:
            parameters = load_parameters(parameters)
        d = parameters['cropping_region']
        g = (d[k] for k in ('top', 'left', 'bottom', 'right'))
        self.__top, self.__left, self.__bottom, self.__right = g
        target_extent = parameters.get('target_extent')
        self.__target_extent = target_extent and tuple(target_extent)
        if type(model) is str:
            model = Model(parameters, model)
        self.__model = model

    def __str__(self) -> str:
        return str(self.__model)

    def detect(self, image: np.array) -> Tuple[Any, float]:
        image = self.preprocess_image(image)
        return self.__model.predict(image)

    def preprocess_image(self, image: np.array) -> np.array:
        image = image[self.__top:self.__bottom, self.__left:self.__right]
        image = change_image_type(image)
        if self.__target_extent is not None:
            image = cv2.resize(image, self.__target_extent, interpolation=cv2.INTER_AREA)
        return image

if __name__ == '__main__':
    import sys
    detector = Detector(sys.argv[1], sys.argv[2])
    image = cv2.imread(sys.argv[3])
    result = detector.detect(image)
    print(*result)
