import cv2
import numpy as np
import os
import requests
import sys
import tempfile
from typing import Callable, Iterable, Optional
try:
    from .m3u8_parser import ObserverFn, Segment, Stream
except ImportError:
    from m3u8_parser import ObserverFn, Segment, Stream

class FrameProvider(Stream):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.__file_path = None
        self.__segment_videos = []
        self.__session = requests.Session()

    def __iter__(self) -> Iterable[np.array]:
        def fn(l):
            segment, video = l
            if video is None:
                video = l[1] = self.__fetch_video_for_segment(segment)
            _, frame = video.read()
            return frame
        return (f for f in map(fn, reversed(self.__segment_videos)) if f is not None)

    def process(self, observer_fn: ObserverFn) -> Optional[float]:
        def fn(segment):
            self.__segment_videos.append([segment, None])
            if observer_fn(self, segment.duration):
                self.__clear_segment_videos()
        return super().process(fn)

    def close(self) -> None:
        super().close()
        self.__clear_segment_videos()
        if self.__file_path:
            os.unlink(self.__file_path)
            self.__file_path = None

    def __fetch_video_for_segment(self, segment: Segment) -> cv2.VideoCapture:
        with self.__session.get(segment.url) as r:
            if not self.__file_path:
                fd, self.__file_path = tempfile.mkstemp()
                os.close(fd)
            with open(self.__file_path, 'wb') as fout:
                fout.write(r.content)
            video = cv2.VideoCapture(self.__file_path, cv2.CAP_MSMF if sys.platform == 'win32' else cv2.CAP_ANY)
            return video

    def __clear_segment_videos(self) -> None:
        for _, video in self.__segment_videos:
            if video:
                video.release()
        self.__segment_videos.clear()
