import cv2
import itertools as it
import json
import logging
import numpy as np
import requests
from typing import Any, Iterable, List, Sequence

# These are stand-ins for mxnet.gluon.Block and mxnet.Context, respectively.
Block = Any
Context = Any

logger = logging.getLogger(__name__)

api_client_id = '3wen8p9e10o33ptgmgogcz7vzuprcq'
api_limit = 100
desired_resolution = 480
maximum_channels = 1000

def change_image_type(image: np.array) -> np.array:
    if image.dtype == np.uint8:
        image = image.astype(np.float32, casting='safe')
        image /= 255.0
    elif image.dtype != np.float32:
        image = image.astype(np.float32, casting='safe')
    return image

def define_network(wants_convolutions: bool, num_outputs: int) -> Block:
    import mxnet as mx
    net = mx.gluon.nn.Sequential()
    with net.name_scope():
        if wants_convolutions:
            net.add(mx.gluon.nn.Conv2D(channels=20, kernel_size=3, activation='relu'))
            net.add(mx.gluon.nn.MaxPool2D(pool_size=2, strides=2))
            net.add(mx.gluon.nn.Conv2D(channels=20, kernel_size=3, activation='relu'))
            net.add(mx.gluon.nn.MaxPool2D(pool_size=2, strides=2))
        net.add(mx.gluon.nn.Dense(512, activation='relu'))
        net.add(mx.gluon.nn.Dense(512, activation='relu'))
        net.add(mx.gluon.nn.Dense(num_outputs))
    return net

def fetch_channel_logins(count: int, game_id: str, token: str) -> List[str]:
    # Fetch channel logins by fetching user IDs and then their logins.
    user_ids = fetch_user_ids(count, game_id, token)
    user_logins = fetch_logins_for_ids(user_ids, token)
    count = len(user_logins)
    message = f'selected {count} channel{"" if count == 1 else "s"}:  {" ".join(user_logins)}'
    logger.info(message)
    return user_logins

def fetch_logins_for_ids(user_ids: Sequence[str], token: str) -> List[str]:
    url = 'https://api.twitch.tv/helix/users'
    headers = {'Client-ID': api_client_id, 'Authorization': f'Bearer {token}'}
    def fn(k: int):
        params = [('id', n) for n in user_ids[k:k + api_limit]]
        if params:
            with requests.get(url, params=params, headers=headers) as r:
                response = r.json()
            return [d['login'] for d in response['data']]
        return None
    g = (fn(i) for i in it.count(0, api_limit))
    g = it.takewhile(lambda l: l, g)
    return list(it.chain.from_iterable(g))

def fetch_user_ids(count: int, game_id: str, token: str) -> List[str]:
    url = 'https://api.twitch.tv/helix/streams'
    headers = {'Client-ID': api_client_id, 'Authorization': f'Bearer {token}'}
    params = {'game_id': game_id, 'first': min(count, api_limit)}
    user_ids = set()
    while len(user_ids) < count:
        with requests.get(url, params=params, headers=headers) as r:
            response = r.json()
        if 'cursor' not in response['pagination']:
            break
        params['after'] = response['pagination']['cursor']
        user_ids.update(d['user_id'] for d in response['data'])
    return list(user_ids)[:count]

def fetch_vod_ids(count: int, game_id: int, token: str) -> List[str]:
    url = 'https://api.twitch.tv/helix/videos'
    headers = {'Client-ID': api_client_id, 'Authorization': f'Bearer {token}'}
    params = {'game_id': game_id, 'first': min(count, api_limit)}
    vod_ids = set()
    while len(vod_ids) < count:
        with requests.get(url, params=params, headers=headers) as r:
            response = r.json()
        params['after'] = response['pagination']['cursor']
        vod_ids.update(d['id'] for d in response['data'])
    return list(vod_ids)[:count]

def get_mxnet_context() -> Context:
    import mxnet as mx
    return mx.gpu() if mx.context.num_gpus() > 0 else mx.cpu()
    
def load_parameters(parameter_file_path: str) -> dict:
    with open(parameter_file_path) as fin:
        return json.load(fin)

def resize_image(image: np.array) -> np.array:
    return cv2.resize(image, (852, desired_resolution), interpolation=cv2.INTER_AREA)
