package authentication

import (
	"context"
	"errors"
	"net/http"

	"code.justin.tv/devhub/twitch-e2-ingest/dynamo"
	"code.justin.tv/devhub/twitch-e2-ingest/interpol"
	"code.justin.tv/devhub/twitch-e2-ingest/logger"
	"code.justin.tv/devhub/twitch-e2-ingest/models"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/sse/malachai/pkg/s2s/caller"
	owl "code.justin.tv/web/owl/client"
	"golang.org/x/sync/errgroup"
)

var twitchInternalClients = map[string]bool{
	"3wen8p9e10o33ptgmgogcz7vzuprcq": true, // "Stream Event Recognizer" project
	"zx2adynxicu0x3o84xvt1efjd08a64": true, // Vapour - Legends of Runeterra
}

const identityRetryCount = 5

// Auth is the interface that wraps methods for authentication.
//go:generate go run github.com/vektra/mockery/cmd/mockery -name Auth
type Auth interface {
	ValidateClient(ctx context.Context, token string) (models.AuthResp, error)
	IfWhitelistedClient(ctx context.Context, authResult models.AuthResp, gameID string) error
	IfTrustedSource(ctx context.Context, userIDs []string, authResult models.AuthResp) bool
}

type authImpl struct {
	interpolClient interpol.InterpolClient
	owlClient      owl.Client
	allowlist      dynamo.Allowlist
	logger         logger.Logger
}

// NewClient is to initialize client for authentication
func NewClient(owlHost, interpolHost, s2sCallerName string, allowlist dynamo.Allowlist, l logger.Logger) (Auth, error) {
	rtConfig := &caller.Config{}
	rt, err := caller.NewRoundTripper(s2sCallerName, rtConfig, l)
	if err != nil {
		return nil, err
	}

	conf := twitchclient.ClientConf{
		Host: owlHost,
		RoundTripperWrappers: []func(http.RoundTripper) http.RoundTripper{
			func(inner http.RoundTripper) (outer http.RoundTripper) {
				rt.SetInnerRoundTripper(inner)
				return rt
			},
		},
	}

	owlClient, err := owl.NewClient(conf)
	if err != nil {
		return nil, err
	}

	interpolClient, err := interpol.New(interpolHost)
	if err != nil {
		return nil, err
	}

	return &authImpl{
		owlClient:      owlClient,
		interpolClient: interpolClient,
		allowlist:      allowlist,
		logger:         l,
	}, err
}

func (a *authImpl) ValidateClient(ctx context.Context, token string) (models.AuthResp, error) {
	var authResult *models.AuthResp
	var err error
	for i := 0; i < identityRetryCount; i++ {
		if authResult, err = a.interpolClient.GetClientInfo(token); (err != nil || authResult == nil) && i == identityRetryCount-1 {
			return models.AuthResp{}, err
		} else if err == nil && authResult != nil {
			return *authResult, nil
		}
	}
	return models.AuthResp{}, errors.New("unexpected auth failure")
}

func (a *authImpl) authorizeClient(ctx context.Context, clientID, userID string) error {
	auths, err := a.owlClient.Authorizations(ctx, userID, nil)
	if err != nil {
		a.logger.Error(err)
		return err
	}

	for _, auth := range auths {
		if auth.ClientIDCanonical == clientID {
			return nil
		}
	}

	a.logger.Warn("client: ", clientID, " is not able to publish to user: ", userID)
	return errors.New("client not allowed to publish data for user")
}

func (a *authImpl) IfWhitelistedClient(ctx context.Context, authResult models.AuthResp, gameID string) error {
	isAllowlisted, err := a.allowlist.IsAllowlisted(ctx, gameID, authResult.ClientID)
	if err != nil {
		a.logger.Error("DynamoDB allowlist error: ", err)
		return err
	}
	if !isAllowlisted {
		a.logger.Warn("client: ", authResult.ClientID, " is not allowlisted to publish for game: ", gameID)
		return errors.New("client not whitelisted")
	}

	return nil
}

func (a *authImpl) IfTrustedSource(ctx context.Context, userIDs []string, authResult models.AuthResp) bool {
	// If internal twitch client, allowlist to publish for every user
	if _, ok := twitchInternalClients[authResult.ClientID]; ok {
		return true
	}

	// if this app token is allowed to publish for the user
	if err := a.validateAuthForUsers(ctx, authResult.ClientID, userIDs); err == nil {
		return true
	}
	a.logger.Warn("this app client: ", authResult.ClientID, " is not allowed")
	return false
}

func (a *authImpl) validateAuthForUsers(ctx context.Context, clientID string, userIDs []string) error {
	g, _ := errgroup.WithContext(ctx)
	for _, userID := range userIDs {
		id := userID
		g.Go(func() error {
			return a.authorizeClient(context.Background(), clientID, id)
		})
	}
	if err := g.Wait(); err != nil {
		return errors.New("fail to auth client publish for users")
	}
	return nil
}
