package authentication

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/web/owl/oauth2"

	owlMocks "code.justin.tv/devhub/twitch-e2-ingest/authentication/mocks"
	dynamoMocks "code.justin.tv/devhub/twitch-e2-ingest/dynamo/mocks"
	interpolMocks "code.justin.tv/devhub/twitch-e2-ingest/interpol/mocks"
	loggerMocks "code.justin.tv/devhub/twitch-e2-ingest/logger/mocks"
	"code.justin.tv/devhub/twitch-e2-ingest/models"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

const testCorrectToken = "test_correct_token"
const testOwnerID = "test_owner_id"
const testClientRowID = "test_client_row_id"
const testClientIDCanonical = "lmvvigy8ine487bjl9o1u248o0jur9"

var mockUserAuthResponse = models.AuthResp{
	UserID: "test_user_id",
	Login:  "test_login",
	Scopes: []string{},
}

var mockClientAuthResponse = models.AuthResp{
	ClientID: testClientIDCanonical,
	Scopes:   []string{},
}

var mockAuthorizations = []*oauth2.Authorization{
	{
		Valid:             true,
		ID:                0,
		Scope:             []string{"user_read"},
		OwnerID:           testOwnerID,
		ClientRowID:       testClientRowID,
		ClientIDCanonical: testClientIDCanonical,
		CreatedAt:         time.Now(),
		UpdatedAt:         time.Now(),
		ExpiresIn:         0,
		UUID:              "66624333-a832-420a-92e4-2f8d798f4e23",
	},
}

type AuthTest struct {
	suite.Suite
	testAuthClient     *authImpl
	mockInterpolClient *interpolMocks.InterpolClient
	mockOwlClient      *owlMocks.OwlClient
	mockAllowlist      *dynamoMocks.Allowlist
	mockLoggerClient   *loggerMocks.Logger
}

func (suite *AuthTest) SetupTest() {
	mockInterpolClient := &interpolMocks.InterpolClient{}
	mockOwlClient := &owlMocks.OwlClient{}
	mockLoggerClient := &loggerMocks.Logger{}
	mockAllowlist := &dynamoMocks.Allowlist{}

	suite.mockInterpolClient = mockInterpolClient
	suite.mockOwlClient = mockOwlClient
	suite.mockAllowlist = mockAllowlist
	suite.mockLoggerClient = mockLoggerClient
	suite.testAuthClient = &authImpl{
		interpolClient: mockInterpolClient,
		owlClient:      mockOwlClient,
		allowlist:      mockAllowlist,
		logger:         mockLoggerClient,
	}
}

func (suite *AuthTest) TearDownTest() {
	suite.mockInterpolClient.AssertExpectations(suite.T())
	suite.mockOwlClient.AssertExpectations(suite.T())
	suite.mockAllowlist.AssertExpectations(suite.T())
	suite.mockLoggerClient.AssertExpectations(suite.T())
}

func TestRunProcessorSuite(t *testing.T) {
	suite.Run(t, new(AuthTest))
}

func (suite *AuthTest) TestValidateClient() {
	suite.mockInterpolClient.On("GetClientInfo", testCorrectToken).Return(&mockUserAuthResponse, nil)

	resp, err := suite.testAuthClient.ValidateClient(context.Background(), testCorrectToken)

	suite.mockInterpolClient.AssertNumberOfCalls(suite.T(), "GetClientInfo", 1)
	suite.Nil(err)
	suite.Equal(mockUserAuthResponse, resp)
}

func (suite *AuthTest) TestValidateClient_WillRetry() {
	suite.mockInterpolClient.On("GetClientInfo", testCorrectToken).Return(nil, errors.New("failed to talk to identity"))

	resp, err := suite.testAuthClient.ValidateClient(context.Background(), testCorrectToken)

	suite.mockInterpolClient.AssertNumberOfCalls(suite.T(), "GetClientInfo", identityRetryCount)
	suite.Error(err, "failed to talk to identity")
	suite.Equal(models.AuthResp{}, resp)
}

func (suite *AuthTest) TestValidateClient_NilAuthResp_NilError() {
	suite.mockInterpolClient.On("GetClientInfo", testCorrectToken).Return(nil, nil)

	resp, err := suite.testAuthClient.ValidateClient(context.Background(), testCorrectToken)

	suite.mockInterpolClient.AssertNumberOfCalls(suite.T(), "GetClientInfo", identityRetryCount)
	suite.Nil(err)
	suite.Equal(models.AuthResp{}, resp)
}

func (suite *AuthTest) TestAuthorizeClient() {
	suite.mockOwlClient.On("Authorizations", mock.Anything, testOwnerID, mock.Anything).Return(mockAuthorizations, nil)

	err := suite.testAuthClient.authorizeClient(context.Background(), testClientIDCanonical, testOwnerID)

	suite.NoError(err)
}

func (suite *AuthTest) TestAuthorizeClient_errFromOwl() {
	errMsg := "error from owl client"
	suite.mockOwlClient.On("Authorizations", mock.Anything, testOwnerID, mock.Anything).Return([]*oauth2.Authorization{}, errors.New(errMsg))
	suite.mockLoggerClient.On("Error", mock.Anything).Once()

	err := suite.testAuthClient.authorizeClient(context.Background(), "any client ID", testOwnerID)

	suite.Error(err, errMsg)
}

func (suite *AuthTest) TestAuthorizeClient_unauthorizedClientID() {
	suite.mockOwlClient.On("Authorizations", mock.Anything, testOwnerID, mock.Anything).Return(mockAuthorizations, nil)
	suite.mockLoggerClient.On("Warn", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Once()

	err := suite.testAuthClient.authorizeClient(context.Background(), "any client ID", testOwnerID)

	suite.Error(err, "client not allowed to publish data for user")
}

func (suite *AuthTest) TestAuthorizeClient_twitchInternalClientID() {
	err := suite.testAuthClient.IfTrustedSource(context.Background(), []string{"any user id"}, models.AuthResp{
		ClientID: "3wen8p9e10o33ptgmgogcz7vzuprcq",
		Scopes:   []string{},
	})

	suite.True(err)
}

func (suite *AuthTest) TestIfWhitelistedClient() {
	suite.mockAllowlist.On("IsAllowlisted", mock.Anything, "1234", mockClientAuthResponse.ClientID).Return(true, nil)

	err := suite.testAuthClient.IfWhitelistedClient(context.Background(), mockClientAuthResponse, "1234")

	suite.Nil(err)
}

func (suite *AuthTest) TestIfWhitelistedClient_AllowlistError() {
	suite.mockAllowlist.On("IsAllowlisted", mock.Anything, "1234", mockClientAuthResponse.ClientID).Return(false, errors.New("oops"))
	suite.mockLoggerClient.On("Error", "DynamoDB allowlist error: ", errors.New("oops")).Once()

	err := suite.testAuthClient.IfWhitelistedClient(context.Background(), mockClientAuthResponse, "1234")

	suite.Error(err, "oops")
}

func (suite *AuthTest) TestIfWhitelistedClient_NotAllowlisted() {
	suite.mockAllowlist.On("IsAllowlisted", mock.Anything, "1234", mockClientAuthResponse.ClientID).Return(false, nil)
	suite.mockLoggerClient.On("Warn", "client: ", mockClientAuthResponse.ClientID, " is not allowlisted to publish for game: ", "1234").Once()

	err := suite.testAuthClient.IfWhitelistedClient(context.Background(), mockClientAuthResponse, "1234")

	suite.Error(err, "client not whitelisted")
}

func (suite *AuthTest) TestValidateAuthForUsers_whitelistedClient() {
	suite.mockOwlClient.On("Authorizations", mock.Anything, testOwnerID, mock.Anything).Return(mockAuthorizations, nil)

	err := suite.testAuthClient.validateAuthForUsers(context.Background(), testClientIDCanonical, []string{testOwnerID})

	suite.Nil(err)
}

func (suite *AuthTest) TestValidateAuthForUsers_unwhitelistedClient() {
	suite.mockOwlClient.On("Authorizations", mock.Anything, testOwnerID, mock.Anything).Return(mockAuthorizations, nil)
	suite.mockLoggerClient.On("Warn", mock.Anything, mock.Anything, mock.Anything, mock.Anything)

	err := suite.testAuthClient.validateAuthForUsers(context.Background(), "other client id", []string{testOwnerID})
	suite.Error(err, "fail to auth client publish for users")
}

func (suite *AuthTest) TestIfTrustedSource_whitelistedAndTrustedClient() {
	suite.mockOwlClient.On("Authorizations", mock.Anything, testOwnerID, mock.Anything).Return(mockAuthorizations, nil)

	res := suite.testAuthClient.IfTrustedSource(context.Background(), []string{testOwnerID}, mockClientAuthResponse)

	suite.Equal(true, res)
}

func (suite *AuthTest) TestIfTrustedSource_failedToAuthorizeClient() {
	suite.mockOwlClient.On("Authorizations", mock.Anything, testOwnerID, mock.Anything).Return([]*oauth2.Authorization{}, errors.New("error from owl client"))
	suite.mockLoggerClient.On("Warn", mock.Anything, mock.Anything, mock.Anything)
	suite.mockLoggerClient.On("Error", mock.Anything).Once()

	res := suite.testAuthClient.IfTrustedSource(context.Background(), []string{testOwnerID}, mockClientAuthResponse)
	time.Sleep(5 * time.Second)
	suite.Equal(false, res)
}
