package dynamo

import (
	"context"
	"errors"
	"strconv"
	"testing"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	awsmocks "code.justin.tv/devhub/twitch-e2-ingest/awsmocks/mocks"
	"code.justin.tv/devhub/twitch-e2-ingest/sns"
	"code.justin.tv/devhub/twitch-e2-ingest/sns/mocks"
)

type AllowlistTest struct {
	suite.Suite
	allowlist  *allowlist
	mockDynamo *awsmocks.DynamoDBAPI
	mockSNS    *mocks.Publisher
	gameID     string
}

func (s *AllowlistTest) SetupTest() {
	dynamo := &awsmocks.DynamoDBAPI{}
	sns := &mocks.Publisher{}
	allowlist := &allowlist{
		tableName: "Allowlist",
		dynamo:    dynamo,
		sns:       sns,
	}
	s.allowlist = allowlist
	s.mockDynamo = dynamo
	s.mockSNS = sns
	s.gameID = "123"
}

func (s *AllowlistTest) TearDownTest() {
	s.mockDynamo.AssertExpectations(s.T())
}

func TestRunProcessorSuite(t *testing.T) {
	suite.Run(t, new(AllowlistTest))
}

func (s *AllowlistTest) TestListClients_Some() {
	const createdAtUnix = 1589924987
	s.mockDynamo.On("QueryWithContext", mock.Anything, mock.Anything).Return(&dynamodb.QueryOutput{
		Items: []map[string]*dynamodb.AttributeValue{
			{
				"GameID":    {S: aws.String(s.gameID)},
				"ClientID":  {S: aws.String("test_client_1")},
				"CreatedAt": {N: aws.String(strconv.Itoa(createdAtUnix))},
			},
			{
				"GameID":   {S: aws.String(s.gameID)},
				"ClientID": {S: aws.String("test_client_2")},
			},
		},
	}, nil)

	gameClients, err := s.allowlist.ListClients(context.Background(), &ListClientsRequest{
		GameID: s.gameID,
	})
	s.NoError(err)
	s.Equal(gameClients, []*Client{
		{ID: "test_client_1", CreatedAt: aws.Time(time.Unix(createdAtUnix, 0))},
		{ID: "test_client_2"},
	})
}

func (s *AllowlistTest) TestListClients_None() {
	s.mockDynamo.On("QueryWithContext", mock.Anything, mock.Anything).Return(&dynamodb.QueryOutput{}, nil)

	gameClients, err := s.allowlist.ListClients(context.Background(), &ListClientsRequest{
		GameID: s.gameID,
	})
	s.NoError(err)
	s.Empty(gameClients)
}

func (s *AllowlistTest) TestListClients_FilterByOrg() {
	s.mockDynamo.On("QueryWithContext", mock.Anything, mock.MatchedBy(func(input *dynamodb.QueryInput) bool {
		av := input.ExpressionAttributeValues[":organizationID"]
		return aws.StringValue(av.S) == "test_org" && aws.StringValue(input.FilterExpression) == "OrganizationID = :organizationID"
	})).Return(&dynamodb.QueryOutput{}, nil)

	_, err := s.allowlist.ListClients(context.Background(), &ListClientsRequest{
		GameID:         s.gameID,
		OrganizationID: "test_org",
	})
	s.NoError(err)
}

func (s *AllowlistTest) TestListClients_Error() {
	s.mockDynamo.On("QueryWithContext", mock.Anything, mock.Anything).Return(nil, errors.New("oops"))

	gameClients, err := s.allowlist.ListClients(context.Background(), &ListClientsRequest{
		GameID: s.gameID,
	})
	s.Error(err)
	s.Nil(gameClients)
}

func (s *AllowlistTest) TestIsAllowlisted_True() {
	s.mockDynamo.On("GetItemWithContext", mock.Anything, mock.Anything).Return(&dynamodb.GetItemOutput{
		Item: map[string]*dynamodb.AttributeValue{
			"GameID":    {S: aws.String(s.gameID)},
			"ClientID":  {S: aws.String("test_client")},
			"CreatedAt": {N: aws.String("1589924987")},
		},
	}, nil)

	isAllowlisted, err := s.allowlist.IsAllowlisted(context.Background(), s.gameID, "test_client")
	s.NoError(err)
	s.True(isAllowlisted)
}

func (s *AllowlistTest) TestIsAllowlisted_False() {
	s.mockDynamo.On("GetItemWithContext", mock.Anything, mock.Anything).Return(&dynamodb.GetItemOutput{}, nil)

	isAllowlisted, err := s.allowlist.IsAllowlisted(context.Background(), s.gameID, "test_client")
	s.NoError(err)
	s.False(isAllowlisted)
}

func (s *AllowlistTest) TestIsAllowlisted_Error() {
	s.mockDynamo.On("GetItemWithContext", mock.Anything, mock.Anything).Return(nil, errors.New("oops"))

	isAllowlisted, err := s.allowlist.IsAllowlisted(context.Background(), s.gameID, "test_client")
	s.Error(err)
	s.False(isAllowlisted)
}

func (s *AllowlistTest) TestCreateAllowlistEntry_Success() {
	ts := time.Now().Unix()
	snsTopics := sns.Topics{{ARN: "snsTopic1"}, {ARN: "snsTopic2"}}
	s.mockSNS.On("CreateTopics", mock.Anything, s.gameID, "test_client").Return(snsTopics, nil)
	s.mockDynamo.On("UpdateItemWithContext", mock.Anything, mock.Anything).Return(
		&dynamodb.UpdateItemOutput{
			Attributes: map[string]*dynamodb.AttributeValue{
				"GameID":         {S: aws.String(s.gameID)},
				"ClientID":       {S: aws.String("test_client")},
				"CreatedAt":      {N: aws.String(strconv.FormatInt(ts, 10))},
				"SNSTopics":      {SS: aws.StringSlice([]string{"snsTopic1", "snsTopic2"})},
				"OrganizationID": {S: aws.String("test_org")},
				"Products":       {SS: aws.StringSlice([]string{"Drops 2.0", "Extensions"})},
				"CreatedBy":      {S: aws.String("test_user")},
			},
		}, nil)

	entry, err := s.allowlist.CreateAllowlistEntry(context.Background(), &CreateAllowlistEntryRequest{
		GameID:         s.gameID,
		ClientID:       "test_client",
		OrganizationID: "test_org",
		Products:       []string{"Drops 2.0", "Extensions"},
		CreatedBy:      "test_user",
	})

	s.NoError(err)
	s.Equal(&AllowlistEntry{
		GameID:         s.gameID,
		ClientID:       "test_client",
		CreatedAt:      aws.Time(time.Unix(ts, 0)),
		SNSTopics:      snsTopics,
		OrganizationID: aws.String("test_org"),
		Products:       aws.StringSlice([]string{"Drops 2.0", "Extensions"}),
		CreatedBy:      aws.String("test_user"),
	}, entry)
}

func (s *AllowlistTest) TestCreateAllowlistEntry_AlreadyExistsError() {
	s.mockSNS.On("CreateTopics", mock.Anything, mock.Anything, mock.Anything).Return(sns.Topics{}, nil)
	s.mockDynamo.On("UpdateItemWithContext", mock.Anything, mock.Anything).Return(nil, &dynamodb.ConditionalCheckFailedException{})

	entry, err := s.allowlist.CreateAllowlistEntry(context.Background(), &CreateAllowlistEntryRequest{
		GameID:   "test_game",
		ClientID: "test_client",
	})
	s.Equal(&AlreadyExistsError{gameID: "test_game", clientID: "test_client"}, err)
	s.Nil(entry)

}

func (s *AllowlistTest) TestCreateAllowlistEntry_DynamoDBError() {
	s.mockSNS.On("CreateTopics", mock.Anything, mock.Anything, mock.Anything).Return(sns.Topics{}, nil)
	s.mockDynamo.On("UpdateItemWithContext", mock.Anything, mock.Anything).Return(nil, errors.New("oops"))

	entry, err := s.allowlist.CreateAllowlistEntry(context.Background(), &CreateAllowlistEntryRequest{
		GameID:   s.gameID,
		ClientID: "test_client",
	})
	s.Error(err)
	s.Nil(entry)
}

func (s *AllowlistTest) TestCreateAllowlistEntry_CreateTopicsError() {
	s.mockSNS.On("CreateTopics", mock.Anything, mock.Anything, mock.Anything).Return(nil, errors.New("oops"))
	entry, err := s.allowlist.CreateAllowlistEntry(context.Background(), &CreateAllowlistEntryRequest{
		GameID:   s.gameID,
		ClientID: "test_client",
	})
	s.Error(err)
	s.Nil(entry)

	s.mockDynamo.AssertNotCalled(s.T(), "UpdateItemWithContext")
}

func (s *AllowlistTest) TestUpdateAllowlistEntry_Success() {
	s.mockDynamo.On("UpdateItemWithContext", mock.Anything, &dynamodb.UpdateItemInput{
		TableName: aws.String("Allowlist"),
		Key: map[string]*dynamodb.AttributeValue{
			"GameID":   {S: aws.String("test_game")},
			"ClientID": {S: aws.String("test_client")},
		},
		ConditionExpression: aws.String("attribute_exists(GameID) AND attribute_exists(ClientID)"),
		ReturnValues:        aws.String(dynamodb.ReturnValueAllNew),
		UpdateExpression:    aws.String("SET OrganizationID = :organization_id, Products = :products"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":organization_id": {S: aws.String("test_org")},
			":products":        {SS: aws.StringSlice([]string{"Drops 2.0"})},
		},
	}).Return(&dynamodb.UpdateItemOutput{
		Attributes: map[string]*dynamodb.AttributeValue{
			"GameID":         {S: aws.String("test_game")},
			"ClientID":       {S: aws.String("test_client")},
			"OrganizationID": {S: aws.String("test_org")},
			"Products":       {SS: aws.StringSlice([]string{"Drops 2.0"})},
		},
	}, nil)

	entry, err := s.allowlist.UpdateAllowlistEntry(context.Background(), &UpdateAllowlistEntryRequest{
		GameID:         "test_game",
		ClientID:       "test_client",
		OrganizationID: "test_org",
		Products:       []string{"Drops 2.0"},
	})
	s.NoError(err)
	s.Equal(entry, &AllowlistEntry{
		GameID:         "test_game",
		ClientID:       "test_client",
		OrganizationID: aws.String("test_org"),
		Products:       aws.StringSlice([]string{"Drops 2.0"}),
	})
}

func (s *AllowlistTest) TestUpdateAllowlistEntry_NotFoundError() {
	s.mockDynamo.On("UpdateItemWithContext", mock.Anything, mock.Anything).Return(nil, &dynamodb.ConditionalCheckFailedException{})

	entry, err := s.allowlist.UpdateAllowlistEntry(context.Background(), &UpdateAllowlistEntryRequest{
		GameID:   "test_game",
		ClientID: "test_client",
	})
	s.Equal(&NotFoundError{gameID: "test_game", clientID: "test_client"}, err)
	s.Nil(entry)
}

func (s *AllowlistTest) TestUpdateAllowlistEntry_Error() {
	s.mockDynamo.On("UpdateItemWithContext", mock.Anything, mock.Anything).Return(nil, errors.New("oops"))

	entry, err := s.allowlist.UpdateAllowlistEntry(context.Background(), &UpdateAllowlistEntryRequest{
		GameID:   "test_game",
		ClientID: "test_client",
	})
	s.Error(err)
	s.Nil(entry)
}

func (s *AllowlistTest) TestDeleteAllowlistEntry_Success() {
	s.mockDynamo.On("DeleteItemWithContext", mock.Anything, mock.Anything).Return(&dynamodb.DeleteItemOutput{}, nil)

	err := s.allowlist.DeleteAllowlistEntry(context.Background(), s.gameID, "test_client")
	s.NoError(err)
}

func (s *AllowlistTest) TestDeleteAllowlistEntry_Error() {
	s.mockDynamo.On("DeleteItemWithContext", mock.Anything, mock.Anything).Return(nil, errors.New("oops"))

	err := s.allowlist.DeleteAllowlistEntry(context.Background(), s.gameID, "test_client")
	s.Error(err)
}

func (s *AllowlistTest) TestListEntriesByOrg_Success() {
	s.mockDynamo.On("QueryPagesWithContext", mock.Anything, mock.Anything, mock.Anything).Return(nil)

	_, err := s.allowlist.ListEntriesByOrg(context.Background(), &ListEntriesByOrgRequest{
		OrganizationID: "test_org",
	})
	s.NoError(err)
}

func (s *AllowlistTest) TestListEntriesByOrg_Error() {
	s.mockDynamo.On("QueryPagesWithContext", mock.Anything, mock.Anything, mock.Anything).Return(errors.New("oops"))

	entries, err := s.allowlist.ListEntriesByOrg(context.Background(), &ListEntriesByOrgRequest{
		OrganizationID: "test_org",
	})
	s.Error(err)
	s.Nil(entries)
}

func (s *AllowlistTest) TestListEntries_Success() {
	av := map[string]*dynamodb.AttributeValue{
		"GameID":   {S: aws.String(s.gameID)},
		"ClientID": {S: aws.String("test_client_1")},
	}
	s.mockDynamo.On("ScanPagesWithContext", mock.Anything, &dynamodb.ScanInput{
		ExclusiveStartKey: av,
		Limit:             aws.Int64(1),
		Select:            aws.String(dynamodb.SelectAllAttributes),
		TableName:         aws.String("Allowlist"),
	}, mock.Anything).Return(nil)

	after, err := ToCursor(av)
	s.NoError(err)
	resp, err := s.allowlist.ListEntries(context.Background(), &ListEntriesRequest{
		First: 1,
		After: after,
	})
	s.NoError(err)
	s.NotNil(resp)
}

func (s *AllowlistTest) TestListEntries_WithFilter() {
	s.mockDynamo.On("ScanPagesWithContext", mock.Anything, &dynamodb.ScanInput{
		Select:           aws.String(dynamodb.SelectAllAttributes),
		TableName:        aws.String("Allowlist"),
		FilterExpression: aws.String("GameID IN (:gameID0, :gameID1) OR OrganizationID IN (:organizationID0)"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":gameID0":         {S: aws.String("test_game_0")},
			":gameID1":         {S: aws.String("test_game_1")},
			":organizationID0": {S: aws.String("test_org")},
		},
	}, mock.Anything).Return(nil)

	resp, err := s.allowlist.ListEntries(context.Background(), &ListEntriesRequest{
		Filter: &ListEntriesFilter{
			GameIDs:         []string{"test_game_0", "test_game_1"},
			OrganizationIDs: []string{"test_org"},
		},
	})
	s.NoError(err)
	s.NotNil(resp)
}

func (s *AllowlistTest) TestListEntries_NoCursor() {
	s.mockDynamo.On("ScanPagesWithContext", mock.Anything, &dynamodb.ScanInput{
		Limit:     aws.Int64(1),
		Select:    aws.String(dynamodb.SelectAllAttributes),
		TableName: aws.String("Allowlist"),
	}, mock.Anything).Return(nil)

	resp, err := s.allowlist.ListEntries(context.Background(), &ListEntriesRequest{
		First: 1,
	})
	s.NoError(err)
	s.NotNil(resp)
}

func (s *AllowlistTest) TestListEntries_NoLimit() {
	av := map[string]*dynamodb.AttributeValue{
		"GameID":   {S: aws.String("test_game")},
		"ClientID": {S: aws.String("test_client")},
	}
	s.mockDynamo.On("ScanPagesWithContext", mock.Anything, &dynamodb.ScanInput{
		ExclusiveStartKey: av,
		Select:            aws.String(dynamodb.SelectAllAttributes),
		TableName:         aws.String("Allowlist"),
	}, mock.Anything).Return(nil)

	after, err := ToCursor(av)
	s.NoError(err)

	resp, err := s.allowlist.ListEntries(context.Background(), &ListEntriesRequest{
		After: after,
	})
	s.NoError(err)
	s.NotNil(resp)
}

func (s *AllowlistTest) TestListEntries_Error() {
	s.mockDynamo.On("ScanPagesWithContext", mock.Anything, mock.Anything, mock.Anything).Return(errors.New("oops"))

	resp, err := s.allowlist.ListEntries(context.Background(), &ListEntriesRequest{
		First: 1,
	})
	s.Error(err)
	s.Nil(resp)
}

func TestFilterExprAndExprAttrVals(t *testing.T) {
	testCases := []struct {
		filter              *ListEntriesFilter
		conditionExpression *string
		attributeValues     map[string]*dynamodb.AttributeValue
	}{
		{
			filter:              &ListEntriesFilter{GameIDs: []string{"test_game"}},
			conditionExpression: aws.String("GameID IN (:gameID0)"),
			attributeValues: map[string]*dynamodb.AttributeValue{
				":gameID0": {S: aws.String("test_game")},
			},
		},
		{
			filter:              &ListEntriesFilter{ClientIDs: []string{"test_client"}},
			conditionExpression: aws.String("ClientID IN (:clientID0)"),
			attributeValues: map[string]*dynamodb.AttributeValue{
				":clientID0": {S: aws.String("test_client")},
			},
		},
		{
			filter:              &ListEntriesFilter{OrganizationIDs: []string{"test_org"}},
			conditionExpression: aws.String("OrganizationID IN (:organizationID0)"),
			attributeValues: map[string]*dynamodb.AttributeValue{
				":organizationID0": {S: aws.String("test_org")},
			},
		},
		{
			filter:              &ListEntriesFilter{Product: "Drop Campaign"},
			conditionExpression: aws.String("contains(Products, :product)"),
			attributeValues: map[string]*dynamodb.AttributeValue{
				":product": {S: aws.String("Drop Campaign")},
			},
		},
		{
			filter: &ListEntriesFilter{
				GameIDs:         []string{"test_game_0", "test_game_1"},
				OrganizationIDs: []string{"test_org"},
			},
			conditionExpression: aws.String("GameID IN (:gameID0, :gameID1) OR OrganizationID IN (:organizationID0)"),
			attributeValues: map[string]*dynamodb.AttributeValue{
				":gameID0":         {S: aws.String("test_game_0")},
				":gameID1":         {S: aws.String("test_game_1")},
				":organizationID0": {S: aws.String("test_org")},
			},
		},
		{
			filter: &ListEntriesFilter{
				OrganizationIDs: []string{"test_org_1", "test_org_2"},
				Product:         "Drop Campaign",
			},
			conditionExpression: aws.String("(OrganizationID IN (:organizationID0, :organizationID1)) AND contains(Products, :product)"),
			attributeValues: map[string]*dynamodb.AttributeValue{
				":organizationID0": {S: aws.String("test_org_1")},
				":organizationID1": {S: aws.String("test_org_2")},
				":product":         {S: aws.String("Drop Campaign")},
			},
		},
		{
			filter: &ListEntriesFilter{
				GameIDs:         []string{"test_game_1", "test_game_2"},
				ClientIDs:       []string{"test_client"},
				OrganizationIDs: []string{"test_org_1", "test_org_2"},
				Product:         "Drop Campaign",
			},
			conditionExpression: aws.String("(GameID IN (:gameID0, :gameID1) OR ClientID IN (:clientID0) OR OrganizationID IN (:organizationID0, :organizationID1)) AND contains(Products, :product)"),
			attributeValues: map[string]*dynamodb.AttributeValue{
				":gameID0":         {S: aws.String("test_game_1")},
				":gameID1":         {S: aws.String("test_game_2")},
				":clientID0":       {S: aws.String("test_client")},
				":organizationID0": {S: aws.String("test_org_1")},
				":organizationID1": {S: aws.String("test_org_2")},
				":product":         {S: aws.String("Drop Campaign")},
			},
		},
		{
			filter: &ListEntriesFilter{},
		},
	}

	for _, tc := range testCases {
		condExpr, av := tc.filter.FilterExprAndExprAttrVals()
		assert.Equal(t, tc.conditionExpression, condExpr)
		assert.Equal(t, tc.attributeValues, av)
	}
}

func TestAttributeValuesAndUpdateExpression(t *testing.T) {
	topics := sns.Topics{{ARN: "snsTopic1"}, {ARN: "snsTopic2"}}
	topicsAttrVal := &dynamodb.AttributeValue{SS: aws.StringSlice(topics.ARNs())}

	ts := time.Now()
	tsAttrVal := &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(ts.Unix(), 10))}

	baseExpression := "SET CreatedAt = :created_at, SNSTopics = :sns_topics"

	testCases := []struct {
		request         *CreateAllowlistEntryRequest
		attributeValues map[string]*dynamodb.AttributeValue
		expression      string
	}{
		{
			request: &CreateAllowlistEntryRequest{},
			attributeValues: map[string]*dynamodb.AttributeValue{
				":created_at": tsAttrVal,
				":sns_topics": topicsAttrVal,
			},
			expression: baseExpression,
		},
		{
			request: &CreateAllowlistEntryRequest{
				OrganizationID: "test_org",
			},
			attributeValues: map[string]*dynamodb.AttributeValue{
				":created_at":      tsAttrVal,
				":sns_topics":      topicsAttrVal,
				":organization_id": {S: aws.String("test_org")},
			},
			expression: baseExpression + ", OrganizationID = :organization_id",
		},
		{
			request: &CreateAllowlistEntryRequest{
				CreatedBy: "test_user",
			},
			attributeValues: map[string]*dynamodb.AttributeValue{
				":created_at": tsAttrVal,
				":sns_topics": topicsAttrVal,
				":created_by": {S: aws.String("test_user")},
			},
			expression: baseExpression + ", CreatedBy = :created_by",
		},
		{
			request: &CreateAllowlistEntryRequest{
				OrganizationID: "test_org",
				CreatedBy:      "test_user",
				Products:       []string{"Drops 2.0", "Extensions"},
			},
			attributeValues: map[string]*dynamodb.AttributeValue{
				":created_at":      tsAttrVal,
				":sns_topics":      topicsAttrVal,
				":organization_id": {S: aws.String("test_org")},
				":products":        {SS: aws.StringSlice([]string{"Drops 2.0", "Extensions"})},
				":created_by":      {S: aws.String("test_user")},
			},
			expression: baseExpression + ", OrganizationID = :organization_id, Products = :products, CreatedBy = :created_by",
		},
	}

	for _, tc := range testCases {
		av, exp := tc.request.AttributeValuesAndUpdateExpression(ts, topics)
		assert.Equal(t, tc.attributeValues, av)
		assert.Equal(t, tc.expression, exp)
	}
}

func TestCursor(t *testing.T) {
	t.Run("Cursor can be deserialized into attribute values and vice versa", func(t *testing.T) {
		expected := map[string]*dynamodb.AttributeValue{
			"GameID":   {S: aws.String("test_game")},
			"ClientID": {S: aws.String("test_client")},
		}
		cursor, err := ToCursor(expected)
		assert.NoError(t, err)

		av, err := FromCursor(cursor)
		assert.NoError(t, err)

		assert.Equal(t, expected, av)
	})

	t.Run("Cursor is base64", func(t *testing.T) {
		av, err := FromCursor("fake_cursor")
		assert.Error(t, err)
		assert.Nil(t, av)
	})

	t.Run("Zero values", func(t *testing.T) {
		var av map[string]*dynamodb.AttributeValue
		cursor, err := ToCursor(av)
		assert.NoError(t, err)
		assert.Equal(t, "", cursor)

		av, err = FromCursor("")
		assert.NoError(t, err)
		assert.Nil(t, av)
	})
}
