package dynamo

import (
	"context"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

type ListPagesRequest struct {
	PageSize int64
	Filter   *ListEntriesFilter
}

type ListPagesResponse struct {
	Pages  []*Page
	Total  int64
	Filter *ListEntriesFilter
}

type Page struct {
	EndCursor string
	Count     int64
}

func (a *allowlist) ListPages(ctx context.Context, r *ListPagesRequest) (*ListPagesResponse, error) {
	var (
		pages         []*Page
		paginationErr error
		total         int64
		items         []map[string]*dynamodb.AttributeValue
	)

	var limit *int64
	if r.PageSize > 0 {
		limit = aws.Int64(r.PageSize)
	}

	filterExpression, expAttrVal := r.Filter.FilterExprAndExprAttrVals()
	input := &dynamodb.ScanInput{
		Limit:                     limit,
		TableName:                 aws.String(a.tableName),
		FilterExpression:          filterExpression,
		ProjectionExpression:      aws.String("GameID, ClientID"),
		ExpressionAttributeValues: expAttrVal,
	}
	err := a.dynamo.ScanPagesWithContext(ctx, input, func(output *dynamodb.ScanOutput, lastPage bool) bool {
		// Pages may have fewer than `limit` items. Merge pages so that they each have `limit` items.
		items = append(items, output.Items...)

		if r.PageSize > 0 && int64(len(items)) >= r.PageSize {
			last := items[r.PageSize-1]
			cursor, err := ToCursor(last)
			if err != nil {
				paginationErr = err
				return false
			}

			pages = append(pages, &Page{
				EndCursor: cursor,
				Count:     r.PageSize,
			})
			items = items[r.PageSize:]
		}

		total += aws.Int64Value(output.Count)
		return true
	})

	if l := len(items); l > 0 {
		pages = append(pages, &Page{
			Count: int64(l),
		})
	}

	if err != nil {
		return nil, err
	}

	if paginationErr != nil {
		return nil, err
	}

	return &ListPagesResponse{
		Pages: pages,
		Total: total,
	}, nil
}
