package dynamo

import (
	"encoding/base64"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"

	"code.justin.tv/devhub/twitch-e2-ingest/sns"
)

// An individual item from the DynamoDB table.
// Each field in the struct has a corresponding field in the table.
type AllowlistEntry struct {
	GameID         string     `dynamodbav:",omitempty"`
	ClientID       string     `dynamodbav:",omitempty"`
	CreatedAt      *time.Time `dynamodbav:",omitempty,unixtime"`
	SNSTopics      sns.Topics `dynamodbav:",omitempty,topics"`
	OrganizationID *string    `dynamodbav:",omitempty"`
	Products       []*string  `dynamodb:",omitempty"`
	CreatedBy      *string    `dynamodbav:",omitempty"`
}

type Client struct {
	ID        string
	CreatedAt *time.Time
}

func ToCursor(av map[string]*dynamodb.AttributeValue) (string, error) {
	if av == nil {
		return "", nil
	}

	b, err := json.Marshal(av)
	if err != nil {
		return "", err
	}

	return base64.StdEncoding.EncodeToString(b), nil
}

func FromCursor(c string) (map[string]*dynamodb.AttributeValue, error) {
	if c == "" {
		return nil, nil
	}

	b, err := base64.StdEncoding.DecodeString(c)
	if err != nil {
		return nil, err
	}

	var av map[string]*dynamodb.AttributeValue
	if err := json.Unmarshal(b, &av); err != nil {
		return nil, err
	}

	return av, nil
}

type ListEntriesFilter struct {
	GameIDs         []string
	ClientIDs       []string
	OrganizationIDs []string
	Product         string
}

// Returns a filter expression and expression attribute values of the form
// 		(GameID IN (:gameID0, ...) OR ClientID IN (:clientID0, ...) OR OrganizationID IN (:organizationID0, ...)) AND Product = :product
func (f *ListEntriesFilter) FilterExprAndExprAttrVals() (*string, map[string]*dynamodb.AttributeValue) {
	if f == nil {
		return nil, nil
	}

	exprAttrVal := map[string]*dynamodb.AttributeValue{}
	gameCondition := inConditionExpression("GameID", ":gameID", f.GameIDs, exprAttrVal)
	clientCondition := inConditionExpression("ClientID", ":clientID", f.ClientIDs, exprAttrVal)
	orgCondition := inConditionExpression("OrganizationID", ":organizationID", f.OrganizationIDs, exprAttrVal)
	productCondition := containsConditionExpression("Products", ":product", f.Product, exprAttrVal)

	var conditions []string
	for _, c := range []string{gameCondition, clientCondition, orgCondition} {
		if c != "" {
			conditions = append(conditions, c)
		}
	}

	if len(conditions) == 0 {
		if productCondition != "" {
			return &productCondition, exprAttrVal
		}
		return nil, nil
	}

	conditionExpression := strings.Join(conditions, " OR ")
	if productCondition != "" {
		conditionExpression = fmt.Sprintf("(%s) AND %s", conditionExpression, productCondition)
	}
	return &conditionExpression, exprAttrVal
}

// Returns an `IN` condition expression and modifies the provided expression attribute value.
func inConditionExpression(attributeName string, prefix string, values []string, exprAttrVal map[string]*dynamodb.AttributeValue) string {
	if len(values) == 0 {
		return ""
	}

	var placeholders []string
	for i, v := range values {
		name := fmt.Sprintf("%s%d", prefix, i)
		exprAttrVal[name] = &dynamodb.AttributeValue{S: aws.String(v)}
		placeholders = append(placeholders, name)
	}

	return fmt.Sprintf("%s IN (%s)", attributeName, strings.Join(placeholders, ", "))
}

// Returns a `contains` condition expression and modifies the provided expression attribute value.
func containsConditionExpression(attributeName, placeholder, value string, exprAttrVal map[string]*dynamodb.AttributeValue) string {
	if value == "" {
		return ""
	}

	exprAttrVal[placeholder] = &dynamodb.AttributeValue{S: aws.String(value)}
	return fmt.Sprintf("contains(%s, %s)", attributeName, placeholder)
}
